function p = workingArea(n)
% Author: Paul Mennen (paul@mennen.org)
%
% returns [x y width height] of the primary display useable area
%                            (i.e. not including the taskbar)
% where: x is the distance in pixels from the left edge
%        y is the distance in pixels from the bottom edge
%
% If an argument is supplied and the argument is 4 or less, then the nth
% element of p is returned instead of the 4 element vector.
%
% Suppose you want to create a small figure window at the upper-left corner of the
% screen and another one at the lower-right corner. The obvious way to do this is
% to get the screen size using get(0,'screensize') and use that to position the
% figures. Using that method, one of those figures would cover part of the taskbar
% which would not likely be your intention. To do this properly you need to query
% for the location and size of the screen working area (i.e available area not
% occupied by the task bar). One way to do this is to use this sequence:
% a = figure('WindowState','maximized'); pause(.1); get(a,'outerposition')
%
% One problem with that method is that is hard to be sure how long a pause is
% required. (It doesn't work without the pause.) And even if you immediately
% delete or resize the figure you will still see the screen flash unnecessarily.
% Another problem with that method is that it will only work on Matlab version
% R2020a or newer. For older versions of Matlab, the same thing could be done
% using the JavaFrame undocumented feature, but that won't work with even older
% versions of Matlab or with Octave. This function solves that problem by
% returning the screen working area without needing to create a dummy figure
% window and it works across all Matlab versions (and even Octave).
% 
% Unfortunately this function has only been tested under Windows 10 and probably
% would need substantial modification to work under Linux or Mac OS.
%
% examples: ----------------------------------------------------------------------------------------------------------
% figure('OuterPosition',workingArea);      % creates a figure filling the entire useable screen area
% w = workingArea;
% v = w + [8 8 -16 -16];
% figure('OuterPosition',v);                % similar to above, except it gives you an 8 pixel margin on all sides
% v = [w(1) w(2) 380 250];
% figure('OuterPosition',v);                % creates a 380 by 250 figure in the lower left corner of the working area
% v = [w(1)+w(3)-380 w(2) 380 250];
% figure('OuterPosition',v);                % creates a 380 by 250 figure in the lower right corner of the working area
% v = [w(1) w(2)+w(4)-250 380 250];
% figure('OuterPosition',v);                % creates a 380 by 250 figure in the upper left corner of the working area
% v = [w(1)+w(3)-380 w(2)+w(4)-250 380 250];
% figure('OuterPosition',v);                % creates a 380 by 250 figure in the upper right corner of the working area
% v = [w(1)+w(3)-300 w(2) 300 w(4)];
% figure('OuterPosition',v);                % creates the tallest 300 pixel wide figure as far to the right as possible
% v = [w(1) w(2)+w(4)-200 w(3) 200];
% figure('OuterPosition',v);                % creates the widest 200 pixel high figure as far to the top as possible
%
  ft = inifile('workingArea.txt');
  if length(strfind(ft,' '))            % but it can't have spaces in it
     ft = [tempdir 'workingArea.txt'];  % so use the temp folder instead
     if length(strfind(ft,' '))         % which rarely has spaces in it.
        p = 'cant use temp folder path containing spaces';  return;
     end;
  end;
  f = fopen(ft,'r');  % open powershell results file
  if f<0              % here if the results file doesn't exist (i.e. workingArea hasn't been run)
    fp = strrep(ft,'.txt','.ps1');  g = fopen(fp,'w');   % create the powershell script file
    fprintf(g,'Add-Type -AssemblyName System.Windows.Forms\n');
    fprintf(g,'$Scr = [System.Windows.Forms.Screen]::AllScreens\n');
    fprintf(g,'$Di = $Scr | Where-Object -FilterScript {$_.DeviceName -match ''DISPLAY1''}\n');
    fprintf(g,'$Di.WorkingArea.X, $Di.WorkingArea.Y,\n');
    fprintf(g,'$Di.WorkingArea.Width, $Di.WorkingArea.Height, $Di.Bounds.Height |\n');
    fprintf(g,'Out-File -FilePath %s -Encoding ASCII\n',ft);  fclose(g);
    fb = strrep(ft,'.txt','.bat');  g = fopen(fb,'w');   % create the batch file
    fprintf(g,'Powershell -ExecutionPolicy Bypass %s\n',fp);  fclose(g);
    if atan2(-0,-1)<0  feval('evalc','dos(fb)');  % run the batch file (evalc to run silently)
    else               dos(fb);                   % run the batch file (if compiled)
    end;
    dos(['del ' fp ' 2> nul']);                   % delete powershell script
    dos(['del ' fb ' 2> nul']);                   % delete batch file
    f = fopen(ft,'r');
    if f<0 p = 'workingArea.ps1 failed to execute'; return; end;
  end;
  p = fscanf(f,'%d');  fclose(f);           % read the results of the powershell script
  if length(p)<5 p = 'workingArea failed';  % powershell script should write 5 numbers
  else  p(2) = p(5) - p(2) - p(4);          % convert top edge coordinates to bottom edge
        p = double(p(1:4))';                % return the first 4 elements of p as row vector
        if nargin & n<5 p = p(n); end;
  end;
