function s = screencfg(in) % sets the screen borders to allow
                           % room for the taskbar & title bar.
% If called without an argument, screencfg attempts to determine the
% current screen borders automatically. For Windows based systems it does
% this by calling the auxiliary function workingArea. For non-Windows based
% systems it does this using the WindowState property except for with older
% versions of Matlab which don't have this property. In that case it will
% use a java frame. For non-Windows sytems, in unusual situations (such as
% with a very old Matlab version such as Matlab 6), the automatic procedure
% may fail. In that case, a manual method is used where a figure appears and
% the user must manually maximize the figure using the maximize button in the
% figure title bar.
%
% If you call screencfg with any argument, for example screencfg('manual')
% then the manual method is always used instead of any of the automatic
% methods. One advantage of the manual method is that instead of maximizing
% the figure that appears you can resize and reposition the figure to cover
% the region where you want plots to appear. In that way you can restrict
% the area to be used for plotting, which is not possible with the automatic
% methods (although the need for this is somewhat unusual). Note that even
% when called with an argument, screencfg will still use a previous calculation
% of the screen configuration unless you first call figpos('reset').
%
% screenfig returns a 4 element vector specifying the region where plots can
% be positioned without covering up any portion of the taskbar. It also saves
% this 4 element vector in the 'screenfig' appdata of the root object and also
% saves the vector in a file called screencfg.txt in the plt\ini folder. This
% ensures that the calculations required to determine the screen configuration
% only have to be done once.
%
% If you want screencfg to recalculate the screen configuration (for example
% after the taskbar has been moved, type the command figpos('reset'). This will
% delete the screen configuration information saved by this function as well as
% the information saved by the workingArea.m helper function.

s = getappdata(0,'screencfg');           % has the screen configuration already
                                         % been determined in this Matlab session?
if length(s)==4  return;  end;           % if yes, return that result
f = fopen(inifile('screencfg.txt'),'r'); % if not, has the screen configuration been
if f>0 s = fscanf(f,'%d')';  fclose(f);  % determinted in a previous session?
       if length(s)==4                   % if yes, return that result (from the
         setappdata(0,'screencfg',s);    % screencfg.txt file)
         return;
       end;
end;

a = figure('menubar','none');  w = ' ';               % First lets determine the window border
p = get(a,'position');  u = get(a,'outerposition');   % thickness and title bar height
BT = p(1) - u(1);                                     % border thickness
Title = u(4) - p(4) - 2*BT;                           % title bar height

if ~nargin & ispc      % here for windows systems -------------------------------------
   w = workingArea;
   if ~ischar(w)       % here if workingArea succeeded
      s = w + [BT BT -2*BT -2*BT-Title];
      ssave(s);  delete(a);  return;
   end;
end;

if ~nargin & ischar(w) & atan2(-0,-1)<0 % here for non windows os or workingArea failed (skip for compiled code)
  Mver = version;  Mver = Mver(1:4);
  if Mver(4) == '.'                     % Maintain chronological order in Mver decimal
     Mver(4) = Mver(3);  Mver(3) = '0'; % for example, change 9.8 to 9.08
  end;                                  % for example, 9.11 would remain unchanged
  Mver = sscanf(Mver(1:4),'%f');        % Mver will equal 8.04 for version 8.4 (i.e. R2014b)
  if Mver >= 9.08
    set(a,'WindowState','Max');         % here for R2020a or newer
    for k=1:100 % wait at most 1 second for the maximize command to take effect
      pause(.01); q = get(a,'pos');  if any(p~=q) break; end;
    end;
  elseif exist('isprop') & feval('isprop',a,'Javaframe')
    drawnow; % avoids Java errors
    feval('warning','off','MATLAB:HandleGraphics:ObsoletedProperty:JavaFrame');
    jFig = get(a,'JavaFrame');
    jFig.setMaximized(1);
    for k=1:100 % wait at most 1 second for the maximize command to take effect
      pause(.01); q = get(a,'pos');  if any(p~=q) break; end;
    end;
  else q = p;  % here if we were not able to automatically maximize the figure
  end;

  if p~=q  s = q + [BT-1 BT-1 -2*BT -2*BT];
           ssave(s);  delete(a);  return;
  end;
end;

% Here to do the manual screen configuration method

set(a,'color',[1 1 1]/9,'pos',[250 120 635 465]);
plt('box',[.03 .22 .94 .74],[0 0 0]);
%text(.035,.66,{'Maximize this figure window, or reposition/resize';
%               'it to indicate the screen area that you want Matlab';
%               'to use for plotting. Then press the continue button.'; '';
%               'You will not see this window again unless you enter';
%               'the command:'}, 'color','white','fontsize',18);
text(.048,.63,{'Maximize this figure window, or';
               'reposition/resize it to indicate the screen';
               'area that you want Matlab to use for plotting.'
               'Then press the continue button.'; '';
               'You will not see this window again';
               'unless you enter the command:'}, 'color','white','fontsize',20);
text(.05,.160,'figpos reset','color','yellow','fontsize',28);
text(.40,.154,{'Type this if you move the taskbar.';
               'Or use the functional form figpos(''reset'')'},...
               'fontsize',13,'color',[.4 .4 .4]);
uicontrol('units','norm','pos',[.42 .07 .13 .07],'string','continue','fontsize',13,...
          'user',BT,'callback',@cont);
while 1    s = get(a,'user');
           if length(s)==4 break; else pause(.02); end;
end;
ssave(s); delete(a);

function cont(a,b)
  p = get(gcf,'pos');  BT = get(gcbo,'user');
  set(gcf,'user',p + [BT-1 BT-1 -2*BT -2*BT]);
  
function ssave(s)  % save s to the root appdata as well as to the file screencfg.txt
  setappdata(0,'screencfg',s);
  f = fopen(inifile('screencfg.txt'),'w');
  fprintf(f,'%d ',s);  fclose(f);
