% winplt.m (ver 01Jul23) ---------------------------------------------------
%
% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org
%
% Struggling with Matlab's FFT window display tool (wintool), I found it cumbersome and
% limited. I wanted a way to quickly change window parameters and see the effect on the
% time and frequency shapes and the most common window measures (scalloping and
% processing loss, frequency resolution, and equivalent noise bandwidth). I couldn't
% modify wintool for my taste since most of the code was hidden (pcode). So I wrote
% winplt.m to create a more useable gui for displaying windows. winplt displays traces
% showing the time and frequency domain shapes of 31 different FFT windows and also is a
% tool for designing your own windows by adjusting the kernel coefficients with a slider.
% You can also use winplt's command line interface to return the window time shapes for
% use in your Matlab programs.
%
% For a more complete description of this application, click on the menu box Help tag.
%
% winplt(-1) ---------------------------------------------------------------
%    Displays a list of all windows and their id codes.
%
% winplt(-2) ---------------------------------------------------------------
%    Returns a cell array containing all names of the defined windows.
%
% [name, kernel] = winplt(id) ----------------------------------------------
%    Returns the name of the window associated with that id number (0 to 31)
%    The second output argument (if given) will contain the convolution
%    kernel. For windows not defined by a kernel the Matlab function
%    used to compute it is returned, along with the window parameters.
%
% winplt(id,points) --------------------------------------------------------
%    The first parameter (id) refers to one of the 31 windows listed below,
%    although the last two are user defined windows and can only be used from
%    the gui interface. A column vector is returned (of length points)
%    containing the amplitude corrected window time shape. Amplitude
%    correction means that the average value is one, i.e.:
%    sum(winplt(id,points)) = points
%
% winplt(id,points,1) ------------------------------------------------------
%    Same as above except that the window is scaled using power correction.
%    This means that the average value of the square of the window is one, i.e.:
%    sum(winplt(id,points,1).^2) = points
%
% winplt(id,points,pwr,opt) ------------------------------------------------
%    id:     An index specifying one of the FFT windows
%    points: The number of points in the time window to be generated
%    pwr:    0=amplitude correction, 1=power correction (default = 0)
%    opt:    optional window parameter
%
% winplt (with no arguments) -----------------------------------------------
%   Opens a GUI to display fft window time and frequency shapes.

function [out1,out2] = winplt(id,points,pwr,opt)
VerString = 'Ver 29Jul23';
out1 = 0;
out2 = 0;
switch nargin
case 0,  % no arguments: Create the winplt window
  TRACEc = [0 1 0; 0 1 0; .2 .6 1; .2 .6 1; 1 .2 .2; 1 .2 .2];
  S.tr = pltinit(0,zeros(1,6),'FigName','winplt (FFT window shapes)','Right',[2 4 6],...
    'xy',[0 .143 .093 .787 .798; -1 .006 .723 .077 .217],'Styles','-:-:-:-:',...
    'TIDcback',@TIDbk,'Options','-X-Y','User','--','HelpFile','*/apps/winplt/winplt.htm',...
    'LabelX','Bin number','LabelY',{'Frequency shape (dB)' 'Time shape'},'TRACEc',TRACEc,...
    'addTag',{'defLim' @lim},'Pos',[750 550],'DisTrace',[0 0 1 1 1 1],'AxisLink',0);
  if getappdata(0,'Mver') >= 8.04      % Here for "Late" Matlab versions (R2014b or later)
    plt xright TGLgrid;                % despite right axis, solid grid lines looks ok here
    set(S.tr(2:2:6),'LineStyle','--'); % dotted lines don't look good in "Late" Matlab versions
  end;
  ax = getappdata(gcf,'axis');
  setappdata(ax(1),'hold',1); % prevent cursor movement from updating axis limits
  set(get(ax(2),'ylabel'),'units','norm','pos',[1.05 .3]); % reposition right hand yLabel
  text(.96,-.08,VerString,'units','norm','color',[.7 .7 .7],'fontsize',9);
  % Next, create the window selection popup (large font), the power correction checkbox,
  % then number of bins slider, the hanning number slider (for the RifeVincent windows)
  % the parameter slider, and the default limits button.
  % Note that the last two sliders have the same screen location, but this is by design
  % since only one of them will be set to visible at a time.
  S.sel = plt('pop',[.08 .02 .38 .96],winplt(-2),@winCB,'offset',30,'index',2,'fontsize',18,...
              'hide',[findobj(gcf,'type','uicontrol'); findobj(gcf,'type','line')]);
  S.cor = uic('style','check','pos',[.97 .58 .03 .04],'callb',@winCB,'backgr',get(gcf,'Color'),'val',1);
  S.bin = plt('slider',[.82 .96 .17],[10 1 60 1 500],'# of bins',  @winCB,2);
  S.han = plt('slider',[.63 .96 .17],[ 1 0 10 0  10], 'Hanning ^#',@winCB,2);
  S.par = plt('slider',[.63 .96 .17],[ 0 0  1],' ',@winCB,1,'4 6 3');
  % Next setup the user window string and the label for the power correction checkbox
  % Note that a separate text object was used for the power correction label because
  % the label provided for a checkbox object can't be rotated.
  S.usr = plt('edit',[.13 1.04],0,@winCB,'length',0,'color',[0 1 1],...
              'string','userwin(points,param)','horiz','left');
  S.lbl = text(1.06,.66,'Power corrected','units','norm',...
              'Color',[.7 .8 .95],'Rotation',90,'user',[1 -.5]);
  % Create the string used to display the convolution kernel for windows that are defined
  % by one. Also used to display error messages as well as a brief help string on startup.
  % A popup is used so we can use this string as a menu for modifying the kernel.
  c = {'Move right','Move left','Longer','Shorter','Save','Revert',...
       'Write winplt.mat','Load winplt.mat','-- cancel --'};
  S.ker = plt('pop',[.005 .695 .16 .3],c,@kerCB,'offset',[-.02 8.5],'enable',0,...
              'color',[1 1 1],'fontsize',9,'interp','tex','hide',0,'swap');
  % S.txt(1) & S.txt(5) - Display scalloping loss
  % S.txt(2) & S.txt(6) - Display the 6db bandwidth
  % S.txt(3) & S.txt(7) - Display the equivalent noise bandwidth
  % S.txt(4) & S.txt(8) - Display the worst case processing loss
  for m = 1:8  S.txt(m) = text(0,0,'');  end;  % define info text for two windows
  xp = .01*ones(4,1);  yp = (.962:-.032:.85)';   % x and y positions for info text
  set(S.txt,'fontsize',8,'units','norm','color',TRACEc(1,:),...
       {'pos'},num2cell([[xp; 75*xp] [yp; yp]],2));
  set(S.txt(5:8),'color',TRACEc(3,:),'vis','off');  % text for previous window
  S.adj = 2000;                  % default adjustment increment (.05%)
  set(gcf,'user',S);             % save the handle list for the callbacks
  winCB; lim;                    % plot the default window (Hanning) with the default limits
  ax = getappdata(gcf,'axis'); axr = ax(2);         % find right hand axis
  yt = get(axr,'ytick');  set(axr,'ytick',yt(1:5)); % remove some of the y tick labels
  set(S.ker,'string','  To change windows, right or left click on the window name');
case 1, [out1 out2] = winplt(id,0);  % One argument
otherwise,                           % 2,3,or 4 arguments -------------------------------
  if ~points  % if # of points = 0, just return the window name in Out1 and the amplitude
              % corrected convolution kernel in Out2
    switch id
      case -1, t = winplt(-2); % display a list of the window names and ID numbers
               out1 = prin('ID %s\n',t{:});
      case -2, out1 = {}; % create a cell array containing the window names with their ID numbers
               for k=0:99  t = winplt(k,0); if ~ischar(t) break; end;
               out1 = [out1 {sprintf('%02d: %s',k,t)}];
               end;
      case  0, out1='Boxcar';                       out2=1;
      case  1, out1='Hanning/Rife Vincent';         out2=0;
      case  2, out1='Chebyshev';                    out2={'chebwin',90,'Sidelobe level',[50 150]};
      case  3, out1='Kaiser';                       out2={'kaiser',12.26526,'Beta',[1 30]};
      case  4, out1='Gaussian';                     out2={'gausswin',4.3,'Alpha',[.1 8]};
      case  5, out1='Tukey';                        out2={'tukeywin',.5,'Taper size',[.01 1]};
      case  6, out1='Bartlett';                     out2={'bartlett'};
      case  7, out1='Modified Bartlett-Hanning';    out2={'barthannwin'};
      case  8, out1='Bohman';                       out2={'bohmanwin'};
      case  9, out1='Nuttall';                      out2={'nuttallwin'};
      case 10, out1='Parzen';                       out2={'parzenwin'};
      case 11, out1='Taylor';                       out2={'taylorwin',-50,'Sidelobe level ',[-120 -10],5};
      case 12, out1='Hamming';                      out2=[1 -.428752];
      case 13, out1='Exact Blackman';               out2=[1 -.58201156 .090007307];
      case 14, out1='Blackman';                     out2=[1 -.595238095 .095238095];
      case 15, out1='Blackman (Matlab)';            out2={'blackman','periodic'};
      case 16, out1='Blackman Harris (Matlab)';     out2={'blackmanharris'};
      case 17, out1='Blackman Harris 61dB';         out2=[1 -.54898908 .063135301];
      case 18, out1='Blackman Harris 67dB';         out2=[1 -.58780096 .093589774];
      case 19, out1='Blackman Harris 74dB';         out2=[1 -.61793520 .116766542 -.002275157];
      case 20, out1='B-Harris 92dB (min 4term)';    out2=[1 -.68054355 .196905923 -.016278746];
      case 21, out1='Potter 210';                   out2=[1 -.61129 .11129];
      case 22, out1='Potter 310';                   out2=[1 -.684988 .2027007 -.0177127];
      case 23, out1='FlatTop - Potter 3 term';      out2=[.9990280 -.925752 .35196]; 
      case 24, out1='FlatTop - Potter 4 term';      out2=[.9994484 -.955728 .539289 -.091581]; 
      case 25, out1='FlatTop - Potter 5 term';      out2=[1 -.970179 .653919 -.201947 .017552];
      case 26, out1='FlatTop - Matlab 5 term';      out2={'flattopwin','periodic'};
      case 27, out1='FlatTop - small coef. 5 term'; out2=[1 -.965 .645 -.194 .014];
      case 28, out1='FlatTop - Mennen 6 term';      out2=[1 -.98069 .79548 -.41115 .104466 -.0080777];
      case 29, out1='FlatTop - Mennen 7 term';      out2=[1 -.98927 .85809 -.56266 .24906 -.060084 4.762e-3];
      case 30, out1='adjust kernel';                out2={'adjustK',0,'Parameter',[-100 100],0,0};
      case 31, out1='user';                         out2={'userwin',0,'Parameter',[-100 100],0,0};
      case 32, out1=0; % indicates no more IDs
    end;
    return;
  end;  % end if ~points
  % here if number of points is non-zero: return the time domain window shape in Out1
  if nargin<3 pwr=0; end;   % use amplitude correction if not specified
  if nargin<4 opt=[]; end;  % optional window parameter
  [out1 c] = winplt(id,0);  % get convolution kernel
  if iscell(c) % Here for Matlab computed windows ----------------------------------
    if isempty(opt) & length(c)>1
      opt = c{2};    % If the optional param is missing, use the default value
    end;
    fcn = c{1};
    if exist(fcn)
      switch length(c)                             % get computed time window 
        case 1, out1 = feval(fcn,points);          %   (without option)
        case 2, out1 = feval(fcn,points,opt);      %   (with periodic option)
        case 4, out1 = feval(fcn,points,opt);      %   (with 1 parameters)
        case 5, out1 = feval(fcn,points,c{5},opt); %   (with 2 parameters)
      end;
      if pwr out1 = out1 * sqrt(points/sum(out1.^2));  % make it power corrected
      else   out1 = out1 * points/sum(out1);           % make it amplitude corrected
      end;
    else out1 = ['Function ' c{1} ' undefined (Signal Processing toolbox required)'];
    end; % end if exist(c{1})
  else % here for kernel defined windows ------------------------------------------
    if ~c(1)                       % RifeVincent windows
      if isempty(opt) opt=1; end;  % if # of HANNs not specified, assume 1
      c = RifeV(opt);
    end;
    out1 = timew(c,points);
    if pwr out1 = out1 / sqrt((sum(2*c.^2)-c(1)^2)); end;
  end; % end if iscell(c)
end;   % end switch nargin
% end function winplt

function TIDbk  % TraceID call back
  S = get(gcf,'user');
  v = get(S.tr(3:4),'vis');         % get visibilities of traces 3 and 4
  [a k] = min(cellfun('length',v)); % find shortest string ('on' is shorter than 'off')
  set(S.txt(5:8),'vis',v{k});       % enable text if either 3rd or 4th trace is visible
%end function TIDback

function winCB(i1,i2)  % callback function (for all objects with callbacks)
  S = get(gcf,'user');
  id = plt('pop',S.sel) - 1;
  [out1 c] = winplt(id,0);  % get convolution kernel
  par = 'visOFF';
  han = 'visOFF';
  ker = '';
  opt = [];
  b = plt('slider',S.bin); bins = 2*b;      % get number of bins
  if iscell(c) % Here for Matlab computed windows ----------------------------------
    if length(c)>3                          % Here if there is a window parameter
      par = 'visON';
      if length(findobj(gcf,'string',c{3})) % if the correct slider is already there
            opt = plt('slider',S.par);      % then just get the current slider value
      else  opt = c{2};                     % otherwise initialize the slider to the default
            plt('slider',S.par,'label',c{3});
            plt('slider',S.par,'minmax',[c{4} -1e99 1e99]);
            plt('slider',S.par,'val',opt);
      end;
    end;
  else         % Here for kernel defined windows ------------------------
    if ~c(1) han = 'visON';
             opt = plt('slider',S.han);     % get number of HANNs
             if (opt>b) opt=b; plt('slider',S.han,opt); end;
             c = RifeV(opt);
    end;
    S.c = c;  S.n = 0;  ker = putKern(S);   % save kernel and convert to string
  end;
  set(S.usr,'vis','off');  plt('pop',S.ker,'enable',0);  % not a user window
  plt('slider',S.par,par);
  plt('slider',S.han,han);
  plt('slider',S.han,'minmax',[0 b 0 b]);
  lim;
  sz = 2048;                    % number of points to plot for each window
  pwr = get(S.cor,'value');     % value of checkbox (power or amplitude corrected)
  switch id
    case 30,                          % for adjustable kernel
       S = get(gcf,'user');
       c = S.c;
       n = S.n;                       % coefficient to modify with param slider
       if ~n n=length(c); S.n=n; end; % select first coefficient if we just switched to id 30
       delta = '';
       v = plt('slider',S.par);  va = abs(v);
       plt('slider',S.par,'val',0);
       if  v > 99 S.adj = v;  delta = ['  \Delta=' num2str(v)];
       elseif va==2 | va==20
                  r = 1 + .5*va/(S.adj*sqrt(abs(c(n)))); % here for slider arrow or trough
                  if v<0 r=1/r; end;
                  c(n) = r * c(n);
       elseif va  if va<1e-39 v = 0; end; % here if not (< e-39 is considered to be zero)
                  c(n) = v;
       end;
       S.c = c;
       ker = [putKern(S) delta];
       plt('pop',S.ker,'enable',1);
       ts = timew(c,bins);
       t  = timew(c,sz);
       if pwr t = t * sqrt(sz/sum(t.^2));          % make it power corrected
       else   t = t * sz/sum(t);                   % make it amplitude corrected
       end;
    case 31,  % user window selection
       f = strrep(get(S.usr,'str'),'param',num2str(opt));
       try,   ts = evalUsr(f,bins);
              t  = evalUsr(f,sz);
              if pwr t = t * sqrt(sz/sum(t.^2));  % make it power corrected
              else   t = t * sz/sum(t);           % make it amplitude corrected
              end;
       catch, t = cellstr(lasterr);                   % get error string
              t = char(t(find(cellfun('length',t)))); % remove blank lines
       end;
       set(S.usr,'vis','on');
    otherwise, t  = winplt(id,sz,pwr,opt);   % t = time window
               ts = winplt(id,bins,pwr,opt); % ts = short time window
  end;  % end switch id
  if ischar(t)
     ker = t;  t = ones(sz,1);  ts = zeros(bins,1);  % winplt returned an error message
  end;
  set(S.ker,'string',ker);
  sz2 = sz/2;  binres = bins/sz;
  h = 20*log10(eps+abs(fft([ts; zeros(sz-bins,1)]))); % freq shape
  h = h-h(1);                 % normalize to bin 0
  x = binres*(-sz2:sz2-1)';
  if gcbo==S.cor set(S.tr(2),'x',x,'y',t); return; end; % change in power factor correction
  y = get(S.tr(1:4),'y');     %          previous frequency trace
                              %          previous time trace
                              % previous previous frequency trace
                              % previous previous time trace
  y = [{[h(sz2+1:sz); h(1:sz2)]; t}; y]; % prepend current frequency & time traces (rotate h)
  i = int2str(id); u = [{['Freq' i]; ['Time' i]}; get(S.tr(1:4),'user')];
  set(S.tr(1:6),'x',x,{'y'},y,{'user'},u);
  for r6=1:sz2 if h(r6)<-6 break; end; end; % look for 6dB bandwidth
  h = h(1:1+round(.5/binres));  sloss = max(h)-min(h);
  enbw = sz*sum(t.^2)/(sum(t)^2);  ploss = sloss + 10*log10(enbw);
  if sloss < .1 sloss = sloss*1000; su = ' mdB'; else su = ' dB'; end;
  plt('rename',u); % set the trace IDs to indicate the correct window ID numbers
  s = {'scallop loss' '6dB bandwidth' 'equiv noise BW' 'worst case PL'};
  set(S.txt,{'string'},...
    [prin('%s = %6w%s ~; %s = %4w bins ~; %s = %4w bins ~; %s = %4w dB',...
         s{1},sloss,su,s{2},2*(r6-1)*binres,s{3},enbw,s{4},ploss);
     get(S.txt(1:4),{'string'})]);  % move current window info text to previous
  set(gcf,'user',S);
%end function winCB

function c = RifeV(n) % compute the kernel for the RifeVincent windows
  warning off;                           % ingore inexact nchoosek
  c = zeros(1,n+1);
  for k=0:n
    c(k+1) = (-1)^k * nchoosek(2*n,n-k); % kernel is row 2n Pascal's triangle
  end;
  c = c/c(1);                            % normalized result
%end function RifeV

function out = timew(c,sz)  % compute the length sz time window from kernel c
  out =  sz * real(ifft([c zeros(1,sz-2*length(c)+1) fliplr(c(2:end))]))';
  % x = (0:sz-1)'/sz;       % alternative version without ifft
  % out = 0*x + c(1);
  % for k = 2:length(c)
  %   out = out + 2*c(k)*cos(2*pi*(k-1)*x);
  % end;
% end function timew

function kerCB
  S = get(gcf,'user');
  l = length(S.c);  c = S.c;
  f = inifile('winplt.mat');
  switch get(S.ker,'string')
    case 'Move right',       S.n = S.n+1; if S.n>l S.n=1; end;
    case 'Move left',        S.n = S.n-1; if S.n<1 S.n=l; end;
    case 'Longer',           c = [c -c(l)];
    case 'Shorter',          if l>1 c(l) = [];   S.n = min(S.n,l-1); end;
    case 'Save',             set(S.lbl,'user',c);
    case 'Revert',           c = get(S.lbl,'user');
    case 'Write winplt.mat', save(f,'c');
    case 'Load winplt.mat',  load(f);
  end; % end switch
  plt('pop',S.ker,'index',1); % so that left click always selects the 'Move left' choice
  set(S.ker,'string',putKern(S));
  if ~isequal(c,S.c)  S.c = c;
                      if S.n > length(c)  S.n = 1;  end;
  end;
  set(gcf,'user',S);
  winCB;
%end function kerCB

function t = putKern(S)
  n = S.n;
  c = S.c;
  t = prin('   {%7w   }<--kernel',c);
  if n % bold face & put parenthesis around the nth coefficient
    p = findstr('   ',t); q = p(n+1); p = p(n);
    t = [t(1:p) ' ({\bf' t(p+1:q) '}) ' t(q+1:end)];
  end;
%end function putKern

function lim(a,b)
  S = get(gcf,'user');  ax = getappdata(gcf,'axis');  w = plt('pop',S.sel);
  b = plt('slider',S.bin);  ylim = [-145 3];  e = ylim + 140 - 20*(w>23) + (w<2)/4;
  set(ax,'xlim',[-b b],{'ylim'},{ylim; e/40});  plt('grid',ax(1));

function a = evalUsr(f,pts) % evaluate function f to get column vector of length pts
  a = eval(strrep(f,'points',int2str(pts)));
  a = a(:);
  n = length(a);
  if     n<pts a = [a; zeros(pts-n,1)];  % zero pad if too short
  elseif n>pts a = a(1:pts);             % truncate if too long
  end;
%end function evalUsr
