% psdZoom.m --(version 14Sep23)---------------------------------------------------------------------
%
% Generates some simulated data consisting of shaped random noise summed with three sinusoids at
% various frequencies and amplitudes. This data is used to demonstrate how to display the spectrum
% (power spectral density) with exponential averaging and windowing. Also demonstrates how to use
% frequency translation (zoom), to focus on the power of the analysis in a selected frequency band.
%
% If you call psdZoom with no arguments, the plot will update 100 times and then stop.
% If you call psdZoom with any argument (e.g. 'psdZoom go') then the display will continue to
% update until you hit the stop button.
%
% For a full description of this application, click on the Help tag in the menu box.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function psdZoom(go)               % psdZoom, psdZoom(0), or psdZoom('0') starts stopped
  if ~nargin go=0; end;            % psdZoom(1), psdZoom('1'), or psdZoom('run') starts running
  VerString = 'Ver 14Sep23';
  go = go(1);  go = go~='0' & go;
  S.fmax = 40;                     % analysis frequency range 0 to 40 kHz (sample rate = 102.4 kHz)
  xy = [0   .180 .055 .805 .927;  -2  .007 .087 .057 .150;  % position: main plot & menu box
        216 .148 .005 .016 .030;  214 .170 .005 .085 .030;  % position: x cursor label & lower Xaxis editbox
        206 .263 .005 .085 .030;  215 .788 .005 .016 .030;  % position: upper Xaxis editbox & y cursor label
        213 .810 .005 .085 .030;  204 .903 .005 .085 .030]; % position: lower/upper Yaxis editbox
  if figpos('height')<800 | length(getappdata(0,'pltSmall'))  % check the height of the screen real estate
        p = [10 10 730 680];                                % use a smaller figure for small screen sizes
  else  p = [10 10 850 800];                                % otherwise use the full size figure window
  end;
  % create upper and lower plots
  S.tr = pltinit(0,0,0,0,'Options','-Xlog-Ylog','Ylim',[0 33],'Xlim',[0 S.fmax],'subplot',[50i 50],'pos',p,...
                 'xy',xy,'Xlabel','       kHz','Ylabel',{'dB' '         dB'},'HelpFile','*/apps/psdZoom.htm');
  he = 17.6 / p(4);                                         % height of edit boxes (normalized positions)
  p = {[.005 .889 .090 .08]; [.060 .869 .100 .10];          % positions: % start button, hanning super button
       [.005 .080 .100 .30]; [.005 .025 .100 .30]; [.005 .010 .10 .26];  % zoomFactor/avg/framesize popups
       [.054 .733 .080  he]; [.054 .668 .080  he]; [.054 .603 .080 he];  % sinewave 1/2/3 frequency
       [.054 .705 .080  he]; [.054 .640 .080  he]; [.054 .575 .080 he];  % sinewave 1/2/3 amplitude
       [.005 .820 .043  he]; [.050 .820 .043  he]; [.095 .820 .043 he];  % noise shape parameters
       [.005 .880 .135  he]; [.008 .430 .120  he]};         % positions: % center freq edit object & disableAA checkbox
  btn = findobj(gcf,'style','push');                                     % popups will hide the 4 button group
  nlbl = {'noise shape' [.13 .023i]};  zlbl = {'CenterFreq (Hz)' [.135 .023i]}; % edit box labels
  S.go  = plt('pop', p{1},{'start' 'stop'},{@ss 0},'swap',0);            % create start button (super button popup)
  S.han = plt('pop', p{2},{'boxcar' 'hanning' 'hannP'},'swap','black');  % create hanning button (super button popup)
  S.zf  = plt('pop', p{3},2.^(0:9),@cb,   'labely','ZoomFactor','index',6,'hide',btn);    % zoom factor popup
  S.exp = plt('pop', p{4},2.^(0:9),@cb,   'labely','Exp Avg',   'index',6,'hide',btn);    % exponential averaging
  S.sz  = plt('pop', p{5},2.^(7:12),@cbFS,'labely','FrameSize', 'index',3,'hide',btn);    % FrameSize popup
  S.f   = plt('edit',p(6:8), {13609 14202 14231},@cb,'label',{'Fr~q1' 'Fr~q2' 'Fr~q3'});  % 3 sinewave frequencies
  S.a   = plt('edit',p(9:11),{   10     4     2},@cb,'label',{'Am~p1' 'Am~p2' 'Am~p3'});  % 3 sinewave amplitudes
  S.ns  = plt('edit',p(12:14),{10 -8 9},@cc,'format',4,'label',{'' nlbl ''});             % 3 noise shape paremeters
  S.zcf = plt('edit',p{15},14000,@cb,'incr',200,'label',zlbl);                            % zoom center frequency
  S.ups = text(.22,-.11,'','units','norm','fontsize',12);                                 % updates/sec readout
  S.bw  = [text(-.13,.97,'','units','norm');  text(-.13,1.12,'','units','norm')];         % bin width readout positions
  set(S.bw,{'color'},{[0 1 0]; [1 0 1]},'horiz','center');                                % bin width readout colors
  S.daa = uic('style','checkbox','string','disable aaFilt','position',p{16},'callback',@cb,'backgr',[.8 .6 .6]);
  text(.64,-.124,VerString,'units','norm','color',[.7 .7 .7],'fontsize',9);
  set(gcf,'user',S);  cc(0);  cbFS;            % save graphics handles & initialize plot
  % plt('pop',S.go,'index',-go-1);             % start running if go is 1
  funcStart({@plt 'pop' S.go 'index' -go-1});  % alternate to above line. Allows return to command prompt
% end function psdZoom

function cbFS                            % callback for the frame size popup
  S = get(gcf,'user');
  sz = 2^(plt('pop',S.sz)+6);            % FFT input buffer size
  fs = S.fmax * 2.56;                    % Sample rate - somewhat higher than the 2*fmax Nyquist rate
  S.bin = 1000* fs/sz;                   % bin width (convert from kHz to Hz)
  S.s = (2*pi/sz) * (0:length(S.rn)-1);  % cylces from 0 to 2pi every sz points
  set(gcf,'user',S);                     % save new values
  n = sz/2.56;                           % # of unaliased points from the fft output (not counting the DC term)
  set(S.tr(1),'Linewidth',0.5);          % indicate that the trace Ydata has been zeroed
  set(S.tr,'x',S.fmax*(0:n)/n,'y',zeros(1,n+1));  cb;  % update x axes, zero out y axes, recompute zoom data
% end function cbFS

function cc(a)                                % call back for the 3 noise parameters (numerator polynomial)
  S = get(gcf,'user');  bs = 2^22;            % buffer size = 4 megaBytes
  if getappdata(0,'Mver')<7  bs = bs/4; end;  % use only a 1 megaByte buffer for the older matlab ver6
  S.rn = filter(plt('edit',S.ns),[10 -14 17 -12 7],rand(1,bs)-.5);  % create data buffer containing shaped random noise
  set(gcf,'user',S);  if ~nargin cb; end;

function cb(a,b)  % callback for the center frequency edit box, checkbox, & the 3 sinewave parameters
  b = [.0161 .0717 .164 .24];
  b = {[b fliplr(b)]                      [.0338 .1478 .2724 .263 .1328 .028] [.06 .266 .4375 .296 .0223 -.0624 -.0202]};
  a = {[1 -1.104 1.546 -.802 .398 -.052]  [1  -.59  .468]                      1                                       };
    % specs for the decimate by 2 filters (above) ---------------------------
                     % Efficiency: 78.125% (i.e. passband yields 100 lines out of 128)
                     % PassBand:   [0, .19531]                Passband Ripple: +/- .020 dB
                     % StopBand:   [.30469, .5]               Stopband Rejection: 70.9 dB
                     % Transfer function:
                     %                      .0161  .0717  .164  .24  .24  .164  .0717  .0161
                     %            H(z) =   --------------------------------------------------
                     %                            1  -1.104  1.546  -.802  .398  -.052
                     %
                     % Efficiency: 39.0625% (i.e. passband yields 50 lines out of 128)
                     % PassBand:   [0, .09766]                Passband Ripple: +/- .0035 dB
                     % StopBand:   [.40234, .5]               Stopband Rejection: 80 dB
                     % Transfer function:
                     %                      .0338  .1478  .2724  .263  .1328  .028
                     %            H(z) =   --------------------------------------------------
                     %                                   1  -.59  .468
                     %
                     % Efficiency: 19.53125% (i.e. passband yields 25 lines out of 128)
                     % PassBand:   [0, .04883]                Passband Ripple: +/- .0046 dB
                     % StopBand:   [.45117, .5]               Stopband Rejection: 77.5 dB
                     % Transfer function:
                     %
                     %            H(z) =  .06  .266  .4375  .296  .0223  -.0624  -.0202
                     %
  S = get(gcf,'user');  y = 0;   z = S.tr(2);  n = length(get(z,'x')) - 1;               % z is the zoom trace handle
  zcf = plt('edit',S.zcf);  cfreq = zcf / S.bin;                                         % get zoom center frequency
  for k = 1:3   y = y + plt('edit',S.a(k)) * cos(S.s*(plt('edit',S.f(k))/S.bin));  end;  % compute noise + 3 sinewaves
  y = S.rn + y/(1.28*n);  S.y = y;                                   % save baseband data
  y = 2 * y .* exp(-1j*cfreq*S.s);                                   % undecimated zoom data
  zf = plt('pop',S.zf);                                              % get zoom factor (decimate by 2^zf)
  w =  S.bin / 2^zf;  x = (zcf + (-n:2:n) * w)/1000;  set(z,'x',x);  % w is half the zoom trace bin width (kHz)
  set(S.bw,{'string'},prin('{\\DeltaF: %8w Hz!col}',[S.bin 2*w]));   % display bin width for both traces
  set(get(z,'parent'),'xlim',x([1 end]));                            % set the x axis limits for the zoom trace
  AA = ~get(S.daa,'value');                                          % use AA filters if AA is true
  for k = zf:-1:1                                                    % decimate by 2^zf
    kk = min(k,3);  if AA y = filter(b{kk},a{kk},y); end;  y = y(1:2:end);
    % prin(1,'numerator:   '); pp(b{kk}); prin(1,'denominator:'); pp(a{kk}); % show the coefficients for each stage
  end;
  S.z = y;  S.new = 1;  set(gcf,'user',S);  % save zoomed data and set the new flag to announce that to the main loop
% end function cb

function ss(in1)                                      % start/stop callback
  S = get(gcf,'user');  tic;
  if strcmp(get(gcbo,'str'),'start') return; end;     % we just hit "stop". Nothing more to do.
  if ~nargin & plt('pop',S.go)>1 return; end;         % already running; no futher action needed.
  if ~isfield(S,'y') return; end;                     % return if simulated data hasn't been created yet
  ups = 1;  tcum = 0;  b = 0;  cntr = 100;            % update display 100 times if starting in the stopped state
  bb = S.tr(1);  zm = S.tr(2);                        % baseband/zoomed traces
  cidl = get(get(bb,'parent'),'user'); q0=-5; q1=35;  % cursorID and Yaxis limits for lower plot
  cidu = get(get(zm,'parent'),'user'); w0=-5; w1=35;  % cursorID and Yaxis limits for upper plot
  while ishandle(S.go) & (plt('pop',S.go)>1 | cntr)   % main display loop
    S = get(gcf,'user');
    if S.new                                                  % here if any of the callbacks been executed
      n = length(get(bb,'x')); szy=2.56*(n-1); i1=0; i2=inf;  % szy/szz are the baseband/zoom frame sizes
      m = length(get(zm,'x')); szz=szy/2;      j1=0; j2=inf;  % n/m are the lengths of the baseband/zoom traces
      ln = length(S.y);  lz = length(S.z);                    % ln/lz are the sizes of the baseband/zoom data sets
      szi = plt('pop',S.sz);  S.new=0;  set(gcf,'user',S);    % get frame size popup index, reset the new flag
    end;
    cntr = max(0, cntr-1);
    i1 = i1 + szy;  i2 = i2 + szy;  if i2 > ln  i1 = 1; i2 = szy; end;  % i1/i2 marks progress thru the baseband data
    j1 = j1 + szz;  j2 = j2 + szz;  if j2 > lz  j1 = 1; j2 = szz; end;  % j1/j2 marks progress thru the zoomed data
    p = fft(S.y(i1:i2));  z = fftshift(fft(S.z(j1:j2)));  han = plt('pop',S.han); % compute FFTs, get han (1 = boxcar)
    if han>1                                        % here if boxcar is not selected
      p = p(1:1+szy/2);                             % keep DC, positive frequencies & fmax (discard negative freq.)
      p = p - (p([end 1:end-1]) + p([2:end 1]))/2;  % compute hanning window (baseband data)
      z = z - (z([end 1:end-1]) + z([2:end 1]))/2;  % comute  hanning window (zoomed data)
      if han>2 c=sqrt(1.5);  p=p/c;  z=z/c; ; end;  % apply power correction factor
    end;
    exi = plt('pop',S.exp);  ex = 2^(1 - exi);  ex1 = 1-ex;  k1 = (szz-m+3)/2;  k2 = k1+m-1;
    p = 20*log10(abs(p(1:n)));  z = 20*log10(abs(z(k1:k2)));          % calculate new PSDs
    if get(bb,'Linewidth')<1  po = p; zo = z; set(bb,'Linewidth',1);  % here if 1st time thru
    else                      po = get(bb,'y');  zo = get(zm,'y');    % here if not. Get old data
    end;
    p = p*ex + po*ex1;  z = z*ex + zo*ex1;  % exponential averaging (combine new data with old data)
    set(bb,'y',p);      set(zm,'y',z);      % save result in line yData
    t2 = toc;
    if t2  tic;  tcum = tcum + t2;  e = .33/ups;  ups = (1-e)*ups + e/t2;  end; % 3 sec time constant for ups readout
    if tcum > 1  tcum = tcum - 1;  b = 1-b;        % update ups readout every second. Alternate color for each update
                 set(S.ups,'color',[b/2 1 b],'string',sprintf('%.2f updates/sec',ups));
    end;
    tc = 4;  tC = 1;                    % use a 4/1 second time constant when narrowing/widening the y axis limits
    g = max(.05,.3 - .025*(szi+exi));   % desired yAxis headroom (smaller for smoother data)
    p0=min(p); p1=max(p);  d=p1-p0; e=d*g; ds=d/60; f=2*e;   new=0;       % check lower yAxis limit for lower plot
    if p0 < q0+ds  q0 = q0 - t2*(q0+e-p0)/tC;                new=1;       % set new if we need to widen the range
    else           hr = p0-q0;  if hr>f q0 = q0 + t2*hr/tc;  new=1; end;  % set new if we need to narrow the range
    end;                                                                  % check upper yAxis limit for lower plot
    if p1 > q1-ds  q1 = q1 + t2*(p1+e-q1)/tC;                new=1;       % set new if we need to widen the range
    else           hr = q1-p1;  if hr>f q1 = q1 - t2*hr/tc;  new=1; end;  % set new if we need to narrow the range
    end;
    if new cur(cidl,'ylim',[q0 q1]); end;                                 % update limits if needed
    z0=min(z); z1=max(z);  d=z1-z0; e=d*g; ds=d/60; f=2*e;   new=0;       % check lower yAxis limit for upper plot
    if z0 < w0+ds  w0 = w0 - t2*(w0+e-z0)/tC;                new=1;       % set new if we need to widen the range
    else           hr = z0-w0;  if hr>f w0 = w0 + t2*hr/tc;  new=1; end;  % set new if we need to narrow the range
    end;                                                                  % check upper yAxis limit for upper plot
    if z1 > w1-ds  w1 = w1 + t2*(z1+e-w1)/tC;                new=1;       % set new if we need to widen the range
    else           hr = w1-z1;  if hr>f w1 = w1 - t2*hr/tc;  new=1; end;  % set new if we need to narrow the range
    end;
    if new cur(cidu,'ylim',[w0 w1]); end;    drawnow;                     % update limits if needed and update plot
  end;                                                   % end of main display loop
  if ishandle(S.go)  S.new=1;  set(gcf,'user',S);  end;  % set the new flag for the next time we hit the start button
% end function ss
