function b = parks(Ord,F,A,W) % Parks-McClellan optimal equiripple FIR filter design
                              % Remez exchange algorithm by Muhammad Ahsan & Tapio Saramaki
%
% This is a simplified version of firpm.m supplied in the signal processing toolbox but it still provides
% the most important features. For most filters of length 35 to 70, parks runs about twice as fast as firpm
% and this speed advantage grows for higher order filters. Here is a comparison of the complexity and size
% of firpm.m (including its private functions) Vs. parks.m (measured with comments removed):
%    firpm:  25 loops   701 lines  12,243 non-blank characters
%    parks:  10 loops   106 lines   3,433 non-blank characters
%
% b = parks(Ord,F,A,W) returns a length Ord+1 linear phase FIR filter (real, symmetric coefficients)
% which has the best approximation to the frequency response described by F and A. F is a vector of
% frequency band edges in pairs, in ascending order between 0 and 1 where 1 corresponds to the Nyquist
% frequency or half the sampling frequency. (i.e. F is twice the angular frequency). A is a real
% vector the same size as F specifying the desired frequency response amplitude. W specifies how
% much emphasis to put on minimizing the error in each band relative to the other bands. W has one
% entry per band, making it half the length of F and A.
%
% The desired response is the line connecting the points (F(k),A(k)) and (F(k+1),A(k+1)) for odd k.
% The bands between F(k) and F(k+1) for even k are transition bands (i.e. don't care regions).
% The maximum error is minimized to this piecewise linear amplitude with transition bands.
%
% For filters with a gain other than zero at Fs/2, e.g., highpass and bandstop filters, the order
% must even. If an odd Ord is specified in this situation, an order Ord+1 filter will be returned
% to insure an even Order.
%
% b = parks(Ord*1i,F,A,W) designs a Hilbert transformer, i.e. a filter with odd symetry.
%
% b = parks(-Ord,F,A,W) designs a differentiator. This also has odd symetry making it similar to the
% Hilbert transformer, but the difference here is that the weighting used is W divided by the angular
% frequency resulting in a filter with a much better fit at low frequencies.
%
% h = parks(55,[0 .3 .4 1],[1 1 0 0],[1 1]);  % create a length 56 lowpass symetric filter
% h = parks(-50,[0 .3 .5 1],[0 1 0 0],[1 1]); % create a length 51 lowpass differentiator (antisymetric)
% h = parks(-33i,[.1 .9],[1 1],1);            % create a length 34 allpass Hilbert transformer (antisymetric)
%
ao = abs(Ord); nc = ao+1;  odd = mod(nc,2);  % nc = number of filter coefficients
sym = (ao==Ord);                             % True for symetric filters
if sym & ~odd & A(end) & F(end)==1 nc=nc+1; odd=1; end; % force odd if F(1) is nonzero

gres = 8;  grd = 0;  gn = 0;  nc2 = floor(nc/2) + (odd & sym);
while gn < 2*nc                                            % loop until grid length is sufficient
  gres = gres*2;  grd(1) = F(1);  df = 1/(nc2*gres);
  if ~sym & (grd(1)<df)  if     F(1)>1e-8  grd(1) = F(1);  % 1st grid point shouldn't be too small
                         elseif df<F(2)    grd(1) = df;
                         else              grd(1) = mean(F(1:2));
                         end;
  end;
  gn = 1;  t = 2;  nb = length(F)/2;  dg = df;
  for k = 1:nb
    fe = F(t);  ggn = grd(gn);  agrd = ggn:df:fe;
    if length(agrd)<11  dg = (fe-ggn)/10;  agrd = ggn:dg:fe;  end;
    grd = [grd agrd+dg];  gm = length(grd);
    if gm<2  gn = gm+1; else grd(gm-1) = fe;  gn = gm;  end;
    if k<nb grd(gn) = F(t+1); end;
    t = t+2;
  end;
  gn = gn-1;  df1 = 1-df;
  if xor(sym,odd) & (grd(gn) > df1)
    if F(end-1) >= df1  grd(gn) = F(end-1);  else gn = gn-1;  end;
  end
  grd = grd(1:gn);
end

s = 1;  t = 2;
for k = 1:nb       % interpolate between A(s) & A(t)
  i = find( grd >= F(s) & grd <= F(t) );  gr = grd(i);  dA = A(t)-A(s);  dF = F(t)-F(s);
  if dF  m = dA/dF;  des(i) = A(s) + m*(gr-F(s));     else  des(i) = mean(A(s:t));  end;
  if real(Ord)<0 & A(t) > 1e-4  wt(i) = 2*W(k) ./ gr;  else  wt(i) = W(k);  end;
  s = s+2;  t = t+2;
end;

gr2 = grd/2;  pgr2 = pi*gr2;  pgrd = pi*grd;  pi2 = 2*pi;  nz = nc2+1;

if sym       if ~odd  v = cos(pgr2);  des = des./v;  wt = wt.*v;  end;
elseif odd            v = sin(pgrd);  des = des./v;  wt = wt.*v;
else                  v = sin(pgr2);  des = des./v;  wt = wt.*v;
end;

ws = warning('off');  % for Matlab ver 6
Ltry = fix([((gn-1)/nc2)*(0:nc2-1)+1 gn]); Ltry = Ltry(1:nz); % begin Remez exchange loop -----
for iter = 1:250 % placing nz indices along the grid (quit after 250 iterations)
  x = cos(pi*grd(Ltry));                              % Lagrange abscissa vector x.
  A = x'*ones(1,nz)-ones(nz,1)*x;  A(eye(nz)==1) = 1;
  ad = prod(A);  ad = 1 ./ (ad * (-2)^(nz-1));        % Lagrange coefficient vector ad
  add = ones(size(ad));  add(2:2:nz) = -add(2:2:nz);
  dev = -(ad*des(Ltry)') / (add*(ad./wt(Ltry))');     % Current value of deviation.
  y = des(Ltry) + dev*add./wt(Ltry);                  % Lagrange ordinate vector y
  xAll = cos(pgrd);                                   % Overall abscissa vector xAll
  errN = zeros(1,gn); errD = errN;   % Initializations of errN and errD.
  for k = 1:nz                       % Intermediate evaluations for the weighted error Werr(k)
    aid = ad(k)./(xAll - x(k));  errD = errD + aid;  errN = errN + y(k)*aid;
  end;
  Werr = (errN./errD - des).*wt;     % Generate the Weighted error function
  devV = ones(size(Ltry));           % Fill in the undefined
  k = 2:2:length(Ltry); devV(k)= -devV(k); Werr(Ltry) = devV * dev; % alternate using dev & -dev
  La1 = find(diff(sign(diff([0 Werr 0]))));  La2 = La1(abs(Werr(La1)) >= abs(dev));
  [u,ind] = max(sparse(1:length(La2), cumsum([1,(Werr(La2(2:end))>=0) ~= (Werr(La2(1:end-1))>=0)]), abs(Werr(La2))));
  Linit = La2(ind);
  if mod(numel(Linit)-nz,2) % if odd difference, discard the 1st or last entry.
    if abs(Werr(Linit(1))) <= abs(Werr(Linit(end)))  Linit(1) = [];  else  Linit(end) = [];  end;
  end;
  while numel(Linit) > nz
    Wreal = abs(Werr(Linit));    Wcomp = max(Wreal(1:end-1), Wreal(2:end));
    if max(abs(Werr(Linit(1))), abs(Werr(Linit(end)))) <= min(Wcomp)  Linit = Linit(2:end-1);
    else                                                              [u,k] = min(Wcomp);  Linit(k:k+1) = [];
    end;
  end
  Lreal = Linit;
  if (Lreal == Ltry) break; else  Ltry=Lreal; end; % Success when the real and trial vectors are the same
end; % end of Remez loop
warning(ws);

nc1 = nc2-1;  nf = 2*nc1 + 1;  df = 1/nf;  k = 1;  x(nz+1) = -2;  a = zeros(1,nc2);
ee = (F(1) | F(end)~=1) & nc2>3;
if ee  cp1 = cos(pi*grd(1));  cpn = cos(pi*grd(gn));  cp = cp1-cpn;  aa = 2/cp;  bb = -(cp1+cpn)/cp; end;
for i=1:nc2  idf = (i-1)*df;   xi = cos(pi2*idf);   xj = x(k);
             if ee  xi = (xi-bb)/aa;  idf = acos(xi)/pi2;  end;
             while  xj-xi > 1e-6  k = k+1;  xj = x(k);  end;
             if abs(xi-xj) <= 1e-6  a(i) = y(k);
             else  c = ad./(cos(pi2*idf)-x(1:nz));  a(i) = (c/sum(c)) * y';
             end;
             k = max(1, k-1);
end;

dden = pi2/nf;  h = zeros(1,nc2);
for i=1:nc2  dnum = dden*(i-1);
             if nc1  h(i) = a(1) + 2*cos(dnum*(1:nc1))*a(2:nc2)'; else  h(i) = a(1); end;
end;
h = [h(1) 2*h(2:end)]/nf;
if ee  q(1) = h(nc2-2) - h(nc2);  p(1) = 2*h(nc2)*bb + h(nc1);  p(2) = 2*aa*h(nc2);
       for i = 2:nc1
         if i==nc1  aa = aa/2;  bb = bb/2;  end;
         p(i+1) = 0;
         j = 1:i;    a(j) = p(j);  p(j) = 2*bb*a(j);  p(2) = p(2) + 2*a(1)*aa;
         j(end)=[];  p(j) = p(j) + q(j) + aa*a(j+1);
         j = j+2;    p(j) = p(j) + aa*a(j-1);
         if i<nc1    j = 1:i;  q(j) = -a(j);  q(1) = q(1) + h(nc2-1-i);  end;
       end;
       h = p;
end;

if Ord<0  h = -h;  end; % flip sign for differentiator
if nc2 < 4  h = [h 0 0]; end;  nn = nz-nc1;  n2 = nc2-nc1+2;
if sym  if odd  b = [2*h(nz-1:-1:nn) 4*h(1)];
        else    b = [h(nc2) h(nz-2:-1:nn)+h(nc2:-1:n2) 2*h(1)+h(2)];
        end;
elseif odd      b = [h(nc2) h(nc1) h(nz-3:-1:nn)-h(nc2:-1:n2+1) 2*h(1)-h(3) 0];
else            b = [h(nc2) h(nz-2:-1:nn)-h(nc2:-1:n2) 2*h(1)-h(2)];
end;
b = fliplr([b (2*sym-1)*b(length(b)-odd:-1:1)])/4;
