% fseries.m -- (version 14Sep23) -----------------------------------------------------------
%
% Demonstrates the ability of a Fourier series (a sum of harmonically related sine waves) to
% approximate periodic functions. In fact there will be a Fourier series that approximates
% any periodic function, although in this application you can only chose one of seven
% different periodic functions (square, triangle, sawtooth, pulse, x^2, abs(sin), and max(0,sin).
%
% The first trace (green) includes only the first term in the series (the lowest frequency sine wave.
% The second trace is sum of the first two terms of the series and the last trace (white) is the
% sum of the first eight terms of the series and is the best approximation of all the traces to the
% desired periodic function.
%
% To make this program more interesting as well as more visually appealing, the amplitudes of the
% traces are continually varied (periodically between plus and minus one) to produce a "real-time"
% moving display. plt is well suited to creating real-time displays, but there are a few concepts to
% learn and this program can help you do that.
%
% To learn more about this application and some of the coding methods used, type "plt help" and
% then click on the "fseries" link in the Signal processing applications section.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function fseries(go)                        % fseries(0) or with no arguments, starts stopped
  if ~nargin go=0; else go = ~~go(1); end;  % fseries(1) or fseries('run') starts running
  p = {[0 .184 .095 .8 .89; -1 .005 .69 .135 .23; -2 .02 .46 .06 .21]; % positions: plot,tIDbox,menubox
       [.040 .860 .100 .10];  [.010 .430 .150 .33];    % positions: start button & Waveform popup
       [.010 .340 .100 .40];  [.010 .266 .100 .30];    % positions: speed and points/cycle popups
       [.010 .180 .100 .20];  [.014 .110 .113 .32]};   % positions: # of cycles popup & box
  cl = [0 1 0; 1 0 1; 0 1 1; 1 0 0; .2 .6 1; 1 .6 .2; .2 .3 1; 1 1 1];             % trace colors
  f = {'square' 'triangle' 'sawtooth' 'pulse' 'x^2     ' 'abs(sin)' 'max(0,sin)'}; % waveforms
  S.tr = pltinit(0,ones(1,8),'Options','Ticks tidcuR -Xlog-Ylog','TRACEc',cl,'Xlim',[0 4*pi],...
                 'Ylim',[-1.05,1.05],'LabelX','Cycles','xy',p{1},'HelpFile','*/apps/fseries.htm');
  S.go  = plt('pop',p{2},{'start' 'stop'},{@gCB 0},'swap',0);
  S.wav = plt('pop',p{3},f,@gCB,'labely','Waveform','interp','tex','offset',-1);
  S.spd = plt('pop',p{4},2.^(0:10),'labely','Speed','index',6,'offset',-1.2);
  S.pts = plt('pop',p{5},25*2.^(0:6),@gCB,'labely','Points/cycle','index',3,'offset',-1.2);
  S.cyc = plt('pop',p{6},2.^(0:4),@gCB,'labely','Cycles','index',2,'offset',-1.2);
  S.ups = text(.13,-.07,'','units','norm','fontsize',13,'user',0);
  plt('box',p{7}); set(S.tr(8),'Linewidth',4);
  set(gcf,'user',S);  gCB;                    % save graphics handles & initialize plot
  % plt('pop',S.go,'index',-go-1);            % start running if go is 1
  funcStart({@plt 'pop' S.go 'index' -go-1}); % alternate to above line. Allows return to command prompt
% end function fseries

function gCB(in1)                                % callback function for all controls
  S = get(gcf,'user');  ru = plt('pop',S.go)>1;  % ru is true when we are  already running
  stop = strcmp(get(gcbo,'str'),'start');        % stop is true if we just pressed the stop button
  if stop | (~nargin & ru)  return; end;         % nothing to do if we just hit stop or plot is moving
  t = {{'1st' '2nd' '3rd' '4th' '5th'  '6th'  '7th'  '8th' };  % all;  for sawtooth & pulse & x^2
       {'2nd' '4th' '6th' '8th' '10th' '12th' '14th' '16th'};  % even; for abs(sin) & max(0,sin)
       {'1st' '3rd' '5th' '7th' '9th'  '11th' '13th' '15th'}}; % odd;  for square & triangle
  g = gcf;  creqSV = get(g,'CloseReq');             % save the close request function
  f0 = 0;  Pts0 = 0;  Ncyc0 = 0;                    % used to detect a parameter change
  b=0;  c=0;  n=0;  tic;                            % start timer
  while ishandle(S.ups) & (plt('pop',S.go)>1 | ~c)  % do the loop once if it's the 1st time thru (c==0)
    if ~mod(n,200)                                  % do this section on every two hundreth update
      b=1-b; set(S.ups,'color',[b 1 .5],'string',sprintf('%.2f updates/sec',n/(toc+eps)));
      f = plt('pop',S.wav);  Pts = str2num(get(S.pts,'string'));    % get function index & Points/cycle
      speed = 2^plt('pop',S.spd)/1e4;  Ncyc = 2^plt('pop',S.cyc)/2; % phase rate of change & # of cycles
      if f ~= f0  |  Pts ~= Pts0 |  Ncyc ~= Ncyc0              % here if a parameter has changed
        f0 = f;  Pts0 = Pts;  Ncyc0 = Ncyc;  c=1;  m=0;  a=1;  % initialize phase counter & amplitude
        x = [0:1/Pts:Ncyc];  v = 0;  y = [];                   % compute new x axis values
        cur(-1,'xylim',[0 Ncyc -1.3 1.3]);             % adjust x axis limits
        oh = f<3;  eo = oh | f>5;                      % oh = odd harmonics. eo = even or odd harmonics
        for k=1:8                                      % compute all 8 harmonics
          p = (eo*k+k-oh)*pi;  e = 2*p*x;  j = k<2;    % j enables the 1st term (k=1) for cases 5,6,7
          switch f
            case 1, w =                       (4/p)            * sin(e); % odd harmonics  - square
            case 2, w =                       (8/p^2)          * cos(e); % odd harmonics  - triangle
            case 3, w =                       (2/p)            * sin(e); % all harmonics  - sawtooth
            case 4, w =                       (2*sin(p/6)/p)   * cos(e); % all harmonics  - pulse
            case 5, w = j/3                 + (4*(-1)^k/p^2)   * cos(e); % all harmonics  - x^2
            case 6, w = j*2/pi              - (64/(4*p^2-1))   * cos(e); % even harmonics - abs(sin)
            case 7, w = j*(1/pi+sin(e/2)/2) - (2/(4*k^2-1)/pi) * cos(e); % even harmonics - max(0,sin)
          end;
          v = v + w;  y(k,:) = v;                                        % accumulate the harmonics
        end;
        for k=1:8 set(S.tr(k),'x',x,'y',a*y(k,:)); end;                  % update trace values
        plt('rename',t{1+oh+eo});                    % rename traces to show which harmonics are being summed
        cur(-1,'setActive',8);  cur(-1,'peakval',2); % put cursor on last trace (white) and on highest peak
      end;                                           % end if f ~= f0
    end;                                             % end if ~mod(n,200)
    for k=1:8 set(S.tr(k),'y',a*y(k,:)); end;        % update trace values
    drawnow;  a = cos(speed*m);                      % refresh  the plot and update the amplitude
    n = n + 1;   m = m + 1;                          % update loop counter and amplitude phase
  end;                                               % end while loop
% end function gCB
