function editz() % editz: A primitive filter design program (ver 14Sep23)------------------
%
% This function demonstrates the usefulness of plt's data editing capability. Two plots
% are created, the lower one showing the poles and zeros of a z-plane transfer function
% and the upper one other showing the magnitude and phase of it's frequency response.
% The frequency response plot automatically updates even while you are dragging a root
% to a new location. At first the updating in real time (i.e. while you are dragging)
% may not seem so important, but when you use the program its becomes clear that this
% allows you to gain a feel for how the root locations effect the frequency response reacts.
%
% For a complete description of this application, click on the "Help" tag in the
% menu box to the left of the pole/zero plot.
%
% The real time motion mentioned above is accomplished by using the "MotionEdit" parameter
% (see line 94).  Normally plt's data editing is initiated when you right click on either
% the x or the y cursor readouts. However when data editing is being used extensively (as in
% the lower plot of this example) it is useful to put it in the "persistent editing mode"
% which is done on line 95. Also, in the following line (line 96) the application data
% variable 'EditCur' is used to change the default size of the cursors used for editing.
%
% Demonstrates the use of the Xstring and Ystring parameters. In the frequency plot, the
% x-cursor edit boxes show the cursor location as a fraction of the sample rate. The
% Xstring parameter is used to show this as an angular measure (in degrees) just to the
% right of the x-cursor readout. Since the DualCur parameter is used, there are two y-cursor
% edit boxes. The first one (green) shows the magnitude response in dB and the second one
% (purple) shows the phase response in degrees. The Ystring parameter is used to show the
% magnitude response in linear form (in percent). Note that after the plot command, the
% Ystring is moved to the left of the plot because by default the Ystring appears in the
% same place as the dual cursor. The alternate location allows room for a multi-line Ystring
% which is generated compliments of prin's cell array output feature. The Xstring parameter
% is also used in the pole/zero plot to show the pole/zero locations in polar form.
%
% The AxisLink parameter is used so that by default the mag/phase axes are controlled separately.
%
% Shows how you can take advantage of both left and right click actions of a button.
% Left-clicking on the "delete p/z" button deletes the root under the cursor as you might expect.
% Right-clicking on this button undoes the previous delete. This is a multi-level undo, so you
% could delete all the roots and then restore them one by one.
%
% Demonstrates the use of the 'Fig' parameter to put two plots in one figure with each plot
% possessing all the features available to any single plot created by plt.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

  ver = 'Ver 14Sep23';
  P = { ...                                    % screen position coordinates -------------------
  [755 720];                                   % P01: figure size
  [-3  .136   .666   .79   .32+.6i];           % P02: frequency response plot position
  [-1  .01    .94    .07   .04    ];           % P03: frequency response plot TraceID
  [-3  .2     .066   .78   .47+.6i];           % P04: pole/zero plot position
  [-1  .04    .27    .08   .06    ];           % P05: pole/zero plot TraceID
  [-2  .008   .080   .05   .135   ];           % P06: pole/zero plot Menu Box
  [.07  .080 .085 .027]; [.07 .115 .085 .027]; % P07: new pole button   P08: new zero button
  [.07  .150 .085 .027]; [.07 .185 .04  .027]; % P09: delete pz button  P10: save button
  [.115 .185 .04  .027]; [.01 .392 .1   .026]; % P11: load button       P12: symmetric check box
  [ 0   .543  1   .054]; [.006 .05 .15      ]; % P13: coefficient box   P14: tolerance slider
  [-3.24  11          ]; [.04   -5.5        ]; % P15: Mag (lin)         P16: quick help
  [.01  0.74          ]; [.01   0.31        ]; % P17: num(z)=           P18: den(z)=
  [.125 -0.4          ]; [.125  -0.9        ]; % P19: num incr          P20: den incr
  [.125 -1.4          ]; [.125  -1.9        ]; % P21: num quant         P22: den quant
  [.11   -3           ]};                      % P23: dc gain

  htxt = {...  % help text
   'Only the roots on or above the x axis are shown.' .22+.92i 2i ...
   '- To move a root:' ...
   '      Drag it with the mouse to the desired location.' ...
   '      The root will snap to the x-axis if you get close.' ...
   '      A zero will snap to the unit circle if you get close.' ...
   '- To add a root, click on the new zero or new pole button.' ...
   '- To remove the cursored root, click on the delete p/z button.' .17+.48i 'color' 505001};
  S.cfg = inifile('editz.mat');             % use plt\ini\editz.mat to save filter coefficients
  if exist(S.cfg) load(S.cfg);              % load editz.mat if it exists
  else            pn = [1 4.3 8 8 4.3 1];   % otherwise use this default numerator polynomial
                  pd = [1 -1 1.5 -.86 .5];  % and this default denominator polynomial
  end;
  z = roots(pn); z=z(find(imag(z)>-1e-5));  % compute initial pole/zero locations
  p = roots(pd); p=p(find(imag(p)>-1e-5));  % (only keep roots on or above the x axis)
  uc = exp((0:.01:1)*pi*1j);                % uc = half unit circle (101 points)
  x = 0:.001:.5;                            % frequency axis
  ys = {'prin("Mag (lin): ~, %6.4f%%",100*min(1,10^(@YVAL/20)))' 'pos' P{15} 'color' 'green' 'fontsize' 9};
  % frequency response plot -----------------------------------------------------------
  S.fr = pltinit(x,x,x,x,'Pos',P{1},'Ylim',[-100 1],'YlimR',[-400 200],'DualCur',2,...
       'TRACEid',{'Mag','Phase'},'LabelY',{'  dB' 'Phase'},'FigBKc',[0 .1 .2],...
       'xy',[P{2}; P{3}],'LabelX','Fraction of sample rate','Options','-Y -H I',...
       'Xstring','sprintf("      %4.2f\\circ",360*@XVAL)','Ystring',ys,'AxisLink',0);
  S.H = exp(pi*get(S.fr(1),'x')*2i); % used for evaluating polynomial around unit circle
  % pole/zero plot ---------------------------------------------------------------------
  S.pz = pltinit('Fig',gcf,z,p,uc,'Styles','nn-','Markers','oxn','MotionEdit',@editPZ,...
       'TRACEc',[0 1 1; 1 1 0; .5 .5 .5],'xy',[P{4}; P{5}; P{6}],'Options','-X-Y-G I B',...
       'TRACEid',['Zeros '; 'Poles ';'circle'],'LabelX','real','LabelY','imag','ENAcur',[1 1 0],...
       'Xstring','sprintf("           (%4.3f, %4.2f\\circ)",abs(@XY),angle(@XY)*180/pi)',...
       'HelpText',htxt,'HelpFile','*/apps/editz.htm');
  setappdata(gca,'moveCBext',1);
  cur 0 Emode Both;                                                % use persistent editing mode
  setappdata(gcf,'EditCur',getappdata(gcf,'EditCur')+6);           % make edit cursor larger
  set(findobj(gcf,'marker','x'),'MarkerSize',10);                  % make poles easier to see
  text(.9,-.1,ver,'units','norm','color',[.5 .5 .5],'fontsize',9); % editz version
  % create 5 buttons. (Left/Right click 3rd button for delete/undelete) ---------------
  uic('Pos',P(7:11),'Str',{'new pole' 'new zero' 'delete p/z' 'save' 'load'},...
      'Callb',{@newPZ @newPZ @curCB @saveload @saveload},'ButtonDown',{'' '' {@curCB 0} '' ''});
  S.pa = [patch(0,0,'k'); patch(0,0,'k')]; % create Xaxis patch and unit circle patch
  set(S.pa,'FaceColor',[0 .15 .15],'EdgeColor',[0 .15 .15]);
  uistack(S.pa,'bottom');                  % move patches to bottom of viewing stack
  if getappdata(0,'Mver') < 7              % here only for Matlab 6 (bug workarounds)
    ch = get(gca,'child');  set(gca,'child',ch([3:end 1 2])); % uistack bug
    set(S.pa,'ButtonDown','cur 0 axisCB;');                   % bug with patch
  end;
  S.nn = 0;  S.dn = 0;                % indicate that no coefficient objects have been created
  plt('box',P{13},20,20,'xlim',0:1);  % dark blue box around numerator/denominator polynomials
  set(text(0,0,'quick help'),'pos',P{16},'color',[0 1 0],'ButtonD','plt helptext on;');
  S.num = text(0,0,'num(z) = ');   S.den = text(0,0,'den(z) = ');
  set([S.num S.den],{'pos'},P(17:18),'color',[1 1 0]); % position the coefficient labels
  cb = {{@incrCB 0} {@incrCB 1} {@quantCB 0} {@quantCB 1} ''};
  lb = {'num i~ncr:' 'den i~ncr:' 'num qu~ant:' 'den qu~ant:' 'DC ga~in:'}';
  [S.nInc S.dInc S.nQ S.dQ S.dc] = dealv( plt('edit',P(19:23),{1000 1000 100 100 0},cb,...
                                               'Label',lb,'Incr',{0 0 inf inf 0}) );
  S.sym  = uic('string','Symmetric','val',1,'style','Check','pos',P{12});
  S.sl = plt('slider',P{14},[.025 0 .05],'Tolerance',@sliderCB,[4 .001],'%1.3f'); % Tolerance slider
  set(gcf,'user',S);                 % save the handle list
  cur(0,'moveCB',@curCB);            % set the move cursor callback function
  set(gcf,'ResizeFcn',{@curCB 0 0}); % set the figure resize function
  sliderCB;  curCB;  curCB(0,0,0,0); % position tolerance patches & update frequency response plot
  plt helptext on;                   % turn on help text
% end function editz

function quantCB(den) % quantize the numerator (den=0) or denominator (den=1) polynomial
  S = get(gcf,'user');
  if den  q = plt('edit',S.dQ);  p = S.den;  % q = quantization level
  else    q = plt('edit',S.nQ);  p = S.num;
  end;
  p = p(2:length(find(strcmp(get(p,'vis'),'on')))); % pick out only the visible edit pseudo objects
  plt('edit',p,'value',num2cell(round(q*plt('edit',p))/q));  % get values, quantize, and put them back
%end function quantCB

function incrCB(den) % change increment parameter for the numerator or denominator polynomial
  S = get(gcf,'user');
  if den  i = plt('edit',S.dInc);  p = S.den;  % i = increment value
  else    i = plt('edit',S.nInc);  p = S.num;
  end;
  if i>0 i = 1/i; end; % negative is precentage increment, otherwise increment is reciprical
  plt('edit',p(2:end),'incr',i);  % change the increment values for the whole polynomial
% end function incrCB

function editPZ(a)              % MotionEdit function
    hact = a{3};                % hact is the handle of the moving edit cursor
    h=get(hact,'user'); h=h{1}; % h is handle of the line being edited
    p = getxy(hact);            % position of edit cursor (diamond)
    q = getxy(h);               % position of poles or zeros being edited
    q(a{9}) = p; setxy(h,q);    % replace its value with the cursor position & update the pz plot
    cur(-2);                    % update FrqResp & Xstring (mag/angle)
% end function editPZ

function newPZ(a,b)                   % add a new pole or zero
  S = get(gcf,'user');
  t = get(gco,'string');              % get button string (pole or zero?)
  u = 1 + (t(5)=='p');                % u=1/2 for new zero/pole
  plt xleft EDIT 5;                   % exit data editing mode
  r = S.pz(u); setxy(r,[0 getxy(r)]); % insert new root (0,0) at the beginning of list
  cur(-2,'setActive',u,1);            % move the cursor to the new root
  plt xleft EDIT 1;                   % and put it back into edit mode
% end function newPZ

function sliderCB()                                  % tolerance slider callback
  S = get(gcf,'user');
  v = plt('slider',S.sl);                            % get tolerance value from slider
  setxy(S.pa(1),[-1-v -1-v 1+v 1+v]+[-v v v -v]*1j); % set patch around x axis
  uc = exp((0:.02:1)*pi*1j);                         % half unit circle
  setxy(S.pa(2),[(1+v)*uc (1-v)*fliplr(uc)]);        % set patch around unit circle (outside/inside)
% end function sliderCB

function curCB(a,b,c,d)                     % pz plot cursor callback (also delete P/Z button)
  S = get(gcf,'user');
  z = getxy(S.pz(1));  p = getxy(S.pz(2));  % get zeros and poles from plot
  al = cur(-2,'getActive');  u = get(gco,'user');
  switch nargin
  case 0, ytol = plt('slider',S.sl);        % here for pz plot cursor callback (snap to tolerance)
          az=find(abs(imag(z))<ytol); ap=find(abs(imag(p))<ytol); % any roots close to the x axis?
          z(az)=real(z(az));          p(ap)=real(p(ap));          % if so, snap to the x axis
          az = find(abs(1-abs(z))<ytol);                          % any zeros close to the unit circle?
          z(az) = exp(angle(z(az))*1j);                           % if so, snap to the unit circle
  case 2, [xy k] = cur(-2,'get');       % delete P/Z button left click - deletes the cursored root
          if     al==1 & length(z)>=k set(gco,'user',[u 1 z(k)]);  z(k) = [];
          elseif al==2 & length(p)>=k set(gco,'user',[u 0 p(k)]);  p(k) = [];
          end;
          plt xleft EDIT 5;                 % exit data editing mode
  case 3, if isempty(u) return; end;        % delete P/Z button right click - MULTI-LEVEL UNDELETE
          if u(end-1)  z = [z u(end)]; else p = [p u(end)]; end; % restore the last deleted root
          set(gco,'user',u(1:end-2));
  end;
  setxy(S.pz(1),z);  setxy(S.pz(2),p);     % in case anything changed
  if nargin==2 cur(-2,'updateN',1); end;
  % update the frequency response plot ------------------------------------------------------------------
  pn = real(poly([z conj(z(find(imag(z)>1e-5)))])); % append complex conjugates, convert to polynomial
  pd = real(poly([p conj(p(find(imag(p)>1e-5)))])); % append complex conjugates, convert to polynomial
  frq(S,pn,pd,z,p);                                 % update frequency response and base workspace
  h = get(gcf,'pos');  w=h(3); h=h(4);  px = 57/w;  % 57 pixels between coefficients
  nn = length(pn);     dn = length(pd);     plt('rename',prin('%2d Zeros ~, %2d Poles',nn-1,dn-1),3);
  nl = length(S.num);  dl = length(S.den);
  if nargin==4                                      % figure size change function
    delete([S.num(2:end) S.den(2:end)]);            % yes, delete & recreate the coefficient edit objects
    S.num = S.num(1); S.den = S.den(1); nl=1; dl=1;  S.nn = 0;  ax = get(S.pz(1),'parent');
    p = get(ax,'pos') .* get(get(ax,'parent'),'pos'); % get size of pole/zero plot (convert normalized to pixels)
    xl = [-1.1 1.1];  dy = p(4) * diff(xl) / (p(3) * 15);  yl = [-dy 14*dy];
    if yl(2) < 1.2  yl = [-.093 1.3];  dx = 0.5 * p(3) * diff(yl) / p(4);  xl = [-dx dx]; end;
    set(ax,'xlim',xl,'ylim',yl);  plt('grid',ax);
  end;
  axes(get(S.num(1),'parent'));         % move to the axis containing the polynomial coefficients
  S.num = dpoly(S.num,pn,nn,nl,px);     % display numerator polynomial
  S.den = dpoly(S.den,pd,dn,dl,-px);    % display denominator polynomial
  if (S.nn ~= nn) | (S.dn ~= dn)               % have we added or removed a root, or resized?
    S.nn = nn;  S.dn = dn;  set(gcf,'user',S); % yes, save S.nn & S.dn and any new edit objects
    incrCB(0); incrCB(1);                      % set the incr parameter of the new edit objects
  end;
  cid = getappdata(gcf,'cid'); cur(cid(1),'update'); % update cursor y-axis positions
  plt helptext;  cur;                                % turn off help text and update cursor
% end function curCB

function S = dpoly(S,pn,nn,nl,px)   % display numerator or denominator polynomial
  clr = [1 .5 .5; 1 1 0];  q = sign(px);  if px<0 py=.31; px=-px; else py=.74; end;
  for k = 1:max(nn,nl-1)
    j = k+1;                        % If there are more edit objects than coefficients,
    if k>nn set(S(j),'vis','off');  % then make the excess edit objects invisible
    else                            % On the other hand, if there are not enough edit
      if j>nl                       % objects, create new ones using purple or yellow text.
        c = clr(1 + mod(floor((k-1)/5),2),:); % Alternate those colors every 5 coefficients
        S(j) = plt('edit',[px*k+.04 py],0,{@coefCB k*q},'color',c); % for easier counting
      end;
      plt('edit',S(j),'val',pn(k),'vis','on'); % set the value and make sure it's visible
    end;
  end;
% end function dpoly

function coefCB(k) % called when kth coefficient is modified (k is negated for denominator)
  S = get(gcf,'user');
  if k>0 & get(S.sym,'val') plt('edit',S.num(S.nn-k+2),'val',plt('edit',S.num(k+1))); end;
  pn = plt('edit',S.num(2:S.nn+1));  z = roots(pn);  % calculate new zeros
  pd = plt('edit',S.den(2:S.dn+1));  p = roots(pd);  % calculate new poles
  k = find(imag(z)<-.001);  z(k) = []; setxy(S.pz(1),z); % don't show roots on lower half plane
  k = find(imag(p)<-.001);  p(k) = []; setxy(S.pz(2),p);
  frq(S,pn,pd,z,p); % update frequency response and base workspace
%end function coefCB

function saveload(a,b)                              % save and load button callbacks
  S = get(gcf,'user');  s = get(gcbo,'str');
  if s(1) == 's'                                    % save callback ---------------
    S = get(gcf,'user');
    pn = plt('edit',S.num(2:S.nn+1));  % get numerator polynomial
    pd = plt('edit',S.den(2:S.dn+1));  % get denominator polynomial
    save(S.cfg,'pn','pd');
    prin(1,'%d numerator and %d denominator coefficients saved\n',S.nn,S.dn);
  else load(S.cfg);                                 % load callback --------------------
       z=roots(pn);  z=z(find(imag(z)>-1e-5));      % plot only upper half of unit circle
       p=roots(pd);  p=p(find(imag(p)>-1e-5));
       setxy(S.pz(1),z);  setxy(S.pz(2),p);  curCB; % update plots
  end;
% end function saveload(a,b)

function frq(S,pn,pd,z,p)                  % update frequency response and base workspace
  pv = conj(S.H);                                        % evaluate num(1/z)/den(1/z) around
  pv = polyval(fliplr(pn),pv) ./ polyval(fliplr(pd),pv); % the unit circle (same as freqz)
  H = 20 * log10(abs(pv));                               % compute frequency response magnitude (dB)
  set(S.fr,{'y'},{H-max(H); angle(pv)*180/pi });         % update frequency response plot (mag/phase)
  plt('edit',S.dc,'val',abs(pv(1)));                     % update DC gain
  assignin('base','z',z);  assignin('base','pn',pn);     % update base workspace
  assignin('base','p',p);  assignin('base','pd',pd);
% end function frq

function c = getxy(h)   % get the x and y coordinates of object h
  c = complex(get(h,'x'),get(h,'y'));

function setxy(h,c)     % set the x and y coordinates of object h
  set(h,'x',real(c),'y',imag(c));
