% afiltS.m:  This program displays the frequency response of traditional analog filters.
% The GUI has been kept simple so that it can be used as an example for how to create a
% graphical user interface. It's used as the 2nd example in the "GUI programming with plt"
% section of the plt help file. afilt.m (in the same folder as afiltS) is a more advanced
% version of this program, which includes several additional features.

function afiltS()
  p = {[.125 .105 .850 .760];  % plot    position
       [.108 .888 .228 .088];  % box     position: Parameter frame
       [.174 .936 .040 .030];  % edit    position: filter order
       [.110 .718 .150 .200];  % popup   position: filter band
       [.314 .756 .050 .200];  % popup   position: # of decades
       [.290 .716 .080 .200];  % popup   position: # of points
       [.350 .946 .150     ];  % slider  position: Passband ripple
       [.510 .946 .150     ];  % slider  position: Stopband ripple
       [.670 .946 .150     ];  % slider  position: Cutoff frequency
       [.830 .946 .150     ]}; % slider  position: frequency 2
  % p = arrange(10); % Used for initial object placement
  band =   {'low pass' 'high pass' 'band pass' 'stop band'};
  ftypes = {'Butter'   'Bessel',   'Cheby1'    'Cheby2'      'Elliptic'};
  S.tr = plt(0,zeros(1,5),'TraceID',ftypes,'Options','logX','xy',p{1},...
             'Ylim',[-80 5],'LabelX','\omega (radians/sec)','LabelY','dB');
  plt('box',            p{2});   % create a frame for the 4 filter parameters
  S.n    = plt('edit',  p{3} ,[5 1 25],   @cb,'BackGr',[1 1 1]/12,'label','Ord~er:');
  S.band = plt('pop',   p{4} ,band,       @cb,'index',3,'swap');
  S.dec  = plt('pop',   p{5} ,1:5,        @cb,'index',3,'label','Decades:','hide');
  S.pts  = plt('pop',   p{6} ,50*2.^(1:5),@cb,'index',2,'label','Points:', 'hide');
  S.Rp   = plt('slider',p{7} ,[ 2  .01   9],'Passband ripple', @cb);
  S.Rs   = plt('slider',p{8} ,[ 40  10 110],'Stopband ripple', @cb);
  S.Wn   = plt('slider',p{9} ,[.1  .01  10],'Cutoff frequency',@cb,5,'%3.2f 6 2');
  S.Wm   = plt('slider',p{10},[.3  .01  10],'frequency 2',     @cb,5,'%3.2f 6 2');
  set(gcf,'user',S);  cb;    % save parameters for cb, and initialize the plot
% end function afiltS

function cb() % callback function for all objects
  S = get(gcf,'user');  N = plt('edit',S.n);              % get handle list and filter order
  dec = plt('pop',S.dec);  pts = 50*2.^plt('pop',S.pts);  % get # of decades and # of points
  bi = plt('pop',S.band);                              % get filter band index (low,high,band,stop)
  Wn  = plt('slider',S.Wn);  Wm = plt('slider',S.Wm);  % get filter frequency & frequency 2
  Rp  = plt('slider',S.Rp);  Rs = plt('slider',S.Rs);  % get passband and stopband ripple
  if bi>2 Wn = [Wn Wm]; v = 'visON'; else v = 'visOFF'; end;
  X = round(log10(mean(Wn)) - dec/2 + .5);  X = logspace(X,X+dec,pts); % X-axis (radians/sec)
  plt('slider',S.Wm,v);       % hide or show frequency 2
  af(bi,N,Wn,Rp,Rs,X,S.tr);   % compute frequency response & set trace data
  cur(-1,'xlim',X([1 end]));  % set Xaxis limits
% end function cb

function af(bi,N,Wn,Rp,Rs,X,g)  % compute analog filter response & set trace data ----------------
                                % bi = 1,2,3,4   for low pass, high pass, pass band, stopband 
  Xj = X*j;  od = rem(N,2);  rs = sqrt(10^(.1*Rs)-1);      % stopband ripple
  rp2 = 10^(.1*Rp)-1;  rp = sqrt(rp2);  rp1 = 1/rp;        % passband ripple
  if bi<3 Wn = Wn(1);                                      % low pass or high pass
  else    bw = diff(Wn);  Wn = sqrt(prod(Wn));  q = Wn/bw; % bandpass or stopband
  end;
  for e=1:5  % e  = 1,2,3,4,5 for butterworth, besself, cheby1, cheby2, ellip
    k = 1;  z = [];
    switch e
    case 1, p = exp(j*(pi*(1:2:N-1)/(2*N) + pi/2));                          % butterworth ---------
            p = [p; conj(p)];  p = p(:);  if od p = [p; -1];  end;
    case 2, m = N:-1:0;  d = N-m;
            p = fac(N+d)./(2.^d.*fac(m).*fac(d));                            % besself ---------
            p = roots(p)/p(end)^(1/N);
    case 3, mu = asinh(rp1)/N;  p = exp(j*(pi*(1:2:2*N-1)/(2*N) + pi/2)).';  % cheby1 ---------
            p = sinh(mu)*real(p) + j*cosh(mu)*imag(p);
            if ~od k = sqrt((1+rp2)); end;
    case 4, mu = asinh(rs)/N;  p = exp(j*(pi*(1:2:2*N-1)/(2*N) + pi/2)).';   % cheby2 ---------
            if od z = j ./ cos([1:2:N-2 N+2:2:2*N-1]*pi/(2*N))';  M=N-1;
            else  z = j ./ cos((1:2:2*N-1)*pi/(2*N))';            M=N;
            end;
            t = [1:M/2; M:-1:M/2+1];  z = z(t(:)); % Organize complex pairs
            p = 1 ./ (sinh(mu)*real(p) + j*cosh(mu)*imag(p));
    case 5, k1 = rp/rs;  k1p = 1-k1^2;  p = [];                              % ellip ---------
            if abs(1-k1p) < eps kr = 0;
            else Ne = N*ellipke(k1^2); kr = Ne/ellipke(k1p); % kr = K(k)/K'(k)
            end;
            set(0,'user',kr);  % find elliptic parameter m so that K(m)/K'(m) = kr
            m = fminsearch(inline('abs(ellipke(min(1,max(x,0)))/ellipke(min(1,max(1-x,0))) - get(0,''user''))'),.5);
            if m>0 & m<1
              em = ellipke(m);  jj = 1-od:2:N-1;  % find zeros on the y axis (imaginary)
              [s,c,d] = ellipj(jj*em/N,m*ones(size(jj)));
              z = j ./(sqrt(m)*s(find(abs(s) > eps)));  z = [z conj(z)];
              set(0,'user',k1p);  setappdata(0,'rp1',rp1);  % save k1p and rp1 for inline function
              r = fminsearch(inline('abs(getappdata(0,''rp1'') - tan(asin(ellipj(u,get(0,''user'')))))'),ellipke(1-m));
              [S,C,D] = ellipj(em*r/Ne,1-m);  p = -complex(d*S*C.*c,s*D)./(1-(d*S).^2);  pj = conj(p);
              if od pj = pj(2:end); else k = sqrt((1+rp2)); end; % remove first pj for odd N, adjust gain for even N
              p = [p pj];
            end;
    end; % end switch e
    [a,b,c,d] = zp2ss(z,p,real(prod(-p)/prod(-z))/k);  % Transform to state-space
    m = size(b,2);  [x,y] = size(c);  ey = eye(y);  v = 0*c;  zy = zeros(y);  zym = zeros(y,m);
    switch bi
      case 1, a = a*Wn;                  b = b*Wn;                                                    % low pass
      case 2, d = d-c/a*b;  c = c/a;     b = -Wn*(a\b);           a =  Wn*inv(a);                     % high pass
      case 3, a = Wn*[a/q ey; -ey zy];   b = Wn*[b/q; zym];       c = [c v];                          % band pass
      case 4, d = d-c/a*b; c = [c/a v];  b = -[Wn/q*(a\b); zym];  a = [Wn/q*inv(a) Wn*ey; -Wn*ey zy]; % band stop
    end;
    den = poly(a);  ld = length(den);  [Z,P,k] = ss2zp(a,b,c,d);  % caclulate num/den polynomials
    ln = length(Z)+1;  num = [zeros(1,ld-ln) k*poly(Z)];
    H = 20*log10(abs(polyval(num,Xj)./polyval(den,Xj)));          % calculate transfer function magnitude (dB)
    set(g(e),'x',X,'y',H);                                        % set trace data
  end;  % end for e = 1:5
% end function af

function n = fac(n)  % Vectorized factorial. Only needed for Matlab 6 since its factorial
  c = cumprod([1 1 2:max(n)]);  n(:) = c(n+1);  % function does not accept vector inputs
