% afiltALT.m -----------------------------------------------------------------
%
% This is an alternate version of afilt.m that differes primarily in the number of traces
% used. While afilt uses 10 traces (5 for magnitude and 5 for phase) afiltALT uses just 5
% traces (on a single axes) to display both magnitude and phase for all 5 filter types.
% The trick to make this work is to use each trace to display both the magnitude and the
% phase information. Although the 10 trace method used by afilt is slightly simpler, this
% alternate version is included since it illustrates a useful technique. Note that the
% tick marks are modified so that they read in degrees in the phase portion of the plot.
% Also the phase portion is highlighted with a gray patch to better separate it visually
% from the magnitude plot. afiltALT is shorter than afilt for 3 reasons:
% 1. It doesn't include the pole/zero plot or the help text
% 2. It doesn't save its state in a configuration file
% 3. It takes advantage of some functions from from the signal processing
%    toolbox (unlike afilt and afiltS)
%
% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function v = afiltALT(arg)
  if nargin % ystring function - displays phase crosshair and value
    [xy k] = cur(-1,'get');                    % get cursor position
    h = cur(-1,'obj');                         % get handles of cursor objects
    hact = h(15);                              % active cursor handle
    lh = get(hact,'user'); y = get(lh{1},'y'); % get line y data
    k = k + floor(length(y)/2) + 1;            % point to phase data
    if k > length(y) v = ''; return; end;      % make sure cursor is in range
    v = prin('%6v\\circ',7.5*y(k)-240);        % scale phase onto magnitude plot
    set(hact,'x',repmat(get(hact,'x'),1,2),... % add phase marker
             'y',[get(hact,'y') y(k)]);
    return;
  end;
  yt1 = -100:20:0;  yt2 = 8:8:56;  b = blanks(27);
  p = {[.140 .100 .840 .770];  % plot    position
       [.108 .888 .228 .088];  % box     position: Parameter frame
       [.174 .936 .040 .030];  % edit    position: filter order
       [.110 .718 .150 .200];  % popup   position: filter band
       [.314 .756 .050 .200];  % popup   position: # of decades
       [.290 .716 .070 .200];  % popup   position: # of points
       [.350 .946 .150     ];  % slider  position: Passband ripple
       [.510 .946 .150     ];  % slider  position: Stopband ripple
       [.670 .946 .150     ];  % slider  position: Cutoff frequency
       [.830 .946 .150     ];  % slider  position: frequency 2
       {-.11 .560          }}; % text    position: eliptic transition ratio
  S.tr = plt(0,[0 0 0 0 0],'xy',p{1},'Options','LogX','ystring','afiltALT(0);',...
    'TraceID',{'Butter' 'Bessel', 'Cheby1'  'Cheby2' 'Elliptic'},'ylim',[-100 60],...
    'LabelX','\omega (radians/sec)','LabelY',[b 'dB' b 'Phase (deg)'],...
    '+ytick',[yt1 yt2],'+yticklabel',num2str([yt1 7.5*yt2-240]'));
  plt('box',p{2});           % create a frame for the 4 filter parameters
  if getappdata(0,'Mver') >= 7
     uistack(patch([1e-9 99 99 1e-9],[5 5 59 59],[1 1 1]/18),'bottom'); % phase plot shading
  end;
  band = {'low pass' 'high pass' 'band pass' 'stop band'};
  S.n    = plt('edit',  p{3}, [5 1 25],   @cb,'BackGr',[1 1 1]/12,'label',{'Order:' .05});
  S.band = plt('pop',   p{4},band,        @cb,'index',3,'swap');
  S.dec  = plt('pop',   p{5},1:5,         @cb,'index',3,'label','Decades:','hide');
  S.pts  = plt('pop',   p{6},50*2.^(1:5), @cb,'index',2,'label','Points:', 'hide');
  a = {[2 .01 9] [40 10 110] [.1 .01 10] [.3 .01 10]};
  t = {'Passband ripple' 'Stopband ripple' 'Cutoff frequency' 'frequency 2'};
  S.sli = plt('slider',p(7:10),a,t,@cb,{'6' '6' '%3.2f 6 2' '%3.2f 6 2'});
  S.etr  = text(p{11}{:},'','units','norm','horiz','center','color',[.2 .6 1]);
  set(gcf,'user',S);
  h = plt('slider',S.sli,'get','obj');  h(:,5)=[];  h = h(:);
  set([h; S.etr],'buttond','plt ColorPick;');
  cb;
  cur -1 updateH -1;  % center cursor
% end function afiltALT

function cb(a,b) % callback function for all objects
  S = get(gcf,'user');  N = plt('edit',S.n);              % get handle list and filter order
  dec = plt('pop',S.dec);  pts = 50*2.^plt('pop',S.pts);  % get # of decades and # of points
  % ty is filter type index, t is filter type name ----================
  bi = plt('pop',S.band);  b = {'low' 'high' 'bandpass' 'stop'}; b=b{bi};
  [Rp Rs Wn Wm] = dealv(plt('slider',S.sli));
  if bi>2 Wn = [Wn Wm]; v = 'visON'; else v = 'visOFF'; end;
  X = round(log10(mean(Wn)) - dec/2 + .5);  X = logspace(X,X+dec,pts); % X-axis (radians/sec)
  plt('slider',S.sli(4),v);                                   % frequency 2 visible for type 3,4
  [B,A] = butter(N,Wn,b,'s');        H1 = frq(B,A,X);         % freq. resp: Butterworth 
  [B,A] = besself(N,Wn(1));          H2 = frq(B,A,X);         % freq. resp: Bessel
  [B,A] = cheby1(N,Rp,Wn,b,'s');     H3 = frq(B,A,X);         % freq. resp: Chebyshev 1
  [B,A] = cheby2(N,Rs,Wn,b,'s');     H4 = frq(B,A,X);         % freq. resp: Chebyshef 2
  [B,A] = ellip(N,Rp,Rs,Wn,b,'s');   H5 = frq(B,A,X);         % freq. resp: Elliptical
  if bi~=1 H2 = H2 + NaN; end;                                % only show Bessel for Lowpass
  set(S.tr,'x',[X NaN X],{'y'},{H1; H2; H3; H4; H5});         % set trace data
  cur(-1,'xlim',X([1 end]));                                  % set Xaxis limits
  h = find(-H5(1:pts) > Rs);                                  % compute Elliptic transition ratio
  if isempty(h)                      h = NaN;   
  elseif (bi-2)*(bi-3)               h = X(h(1))/Wn(1);
  else   h = find(diff([h inf])>1);  h = Wn(1)/X(h(1));
  end;
  set(S.etr,'string',prin('Elliptic ~, transition ~, ratio: ~, %5v',h));
% end function cb

function H = frq(B,A,W) % computes frequency response dB magnitude and phase
  H = freqs(B,A,W);
  H = [20*log10(abs(H)) NaN 32+angle(H)*24/pi];
