% afilt.m -- (version 14Sep23) --------------------------------------------------
%
% Displays the frequency response of the most common traditional analog filters. GUI
% controls are provided to adjust the most important filter parameters (Filter order,
% Cutoff frequency, & Passband/Stopband ripple). A pole/zero plot is also provided for
% the prototype filters (i.e. low pass with a cutoff frequency of 1 radian/sec). Both
% plots (especially the pole/zero plot) are easier to interpret with only one filter
% type enabled. To do this, right click on the "Elliptic" traceID (for example) in the
% black traceID block near the left edge of the figure. That will disable all traces
% on both plots except for the traces associated with the elliptic filter type.
%
% afilt creates these fourteen pseudo objects:
%  1.) a plot
%  2.) a cursor
%  3.) a grid
%  4.) an edit object (filter order)
%  5.) a popup (filter type)
%  6.) a popup (decades to display)
%  7.) a popup (number of points to display)
%  8.) a slider (passband ripple)
%  9.) a slider (stopband ripple)
% 10.) a slider (cutoff frequency)
% 11.) a slider (frequency 2)
% 12.) a secondary plot (pole/zero)
% 13.) a cursor for the 2nd plot
% 14.) a grid for the 2nd plot
%
% The first three three pseudo objects in this list are created by the first call to plt
% and the next  eight pseudo objects are created with eight additional calls to plt. And
% finally one more call to pltinit creates the last 3 pseudo objects.
%
% Although Matlab already has objects with similar names, these pseudo objects are different.
% They provide more utility and options. The pseudo objects 4 thru 7 listed above are grouped
% inside a parameter frame (a grey box implemented by using a Matlab axis object).
%
% There are two alternate versions of this application included in the demo folder (neither
% run by demoplt). The first is named afiltS.m and is a simplfied version of afilt (without
% some of the advanced features). The simplied version is used as the 2nd example in the
% "GUI programming with plt" section of the help file. The other alternate version (called
% afiltALT.m) differs primarily in the number of traces used. While afilt uses 10 traces
% (5 for magnitude and 5 for phase) afiltALT uses just 5 traces (on a single axes) to display
% both magnitude and phase for all 5 filter types. The trick to make this work is to use each
% trace to display both the magnitude and the phase information. Although I eventually
% decided that the 10 trace method was simpler, the alternate version is included since it
% illustrates a useful technique. Note that the tick marks are modified so that they read in
% degrees in the phase portion of the plot. Also the phase portion is highlighted with a
% gray patch to better separate it visually from the magnitude plot. afiltALT is shorter than
% afilt because it doesn't include the pole/zero plot and also (unlike afilt and afiltS) it
% takes advantage of some functions from from the signal processing toolbox.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function afilt()
VerString = 'Ver 14Sep23';
htxt = {'Select the filter order & type' 'in the parameter box above.' '' ...
        'Vary the ripple & frequency' ' parameters using the sliders.' -1.28+.72i 2i ...
        's plane'  .4+.2i 'color' 'yellow' 'fontsize' 20 2i ...
        'Prototype filters' '(low pass with' '1 rad/sec cutoff)' .05+.98i};

  p = {[  0 .090 .105 .573 .760;   % plot 1 position
         -1 .015 .690 .065 .165;   %        traceID box
         -2 .008 .100 .043 .300;   %        menu box
        210 .092 .006 .020 .038;   %        x label
        208 .118 .006 .070 .040;   %        x cur
        209 .526 .006 .020 .038;   %        y label
        206 .550 .006 .070 .040;   %        y cur left
        205 .625 .006 .070 .040;   %        y cur right
        323 -1.10 .55  0    0  ;   %        cursor ID
        324  1.03 .84  0    0  ];  %        right yaxis label
           [.015 .885 .180 .089];  % box    position: Parameter frame
           [.062 .933 .030 .030];  % edit   position: filter order
           [.015 .712 .120 .200];  % popup  position: filter band
           [.174 .752 .050 .200];  % popup  position: # of decades
           [.156 .712 .070 .200];  % popup  position: # of points
           [.210 .946          ];  % slider position: Passband ripple
           [.350 .946          ];  % slider position: Stopband ripple
           [.490 .946          ];  % slider position: Cutoff frequency
           [.630 .946          ];  % slider position: frequency 2
           {-.09 .640          };  % text   position: eliptic transition ratio
           [.790 .931 .075 .045];  % button position: open phase delay plot
           [.790 .882 .075 .042];  % checkbox position: prototype
       [ -3 .740 .050 .245 .814;   % plot 2 position (pole zero plot)
        221 .890 .940 .015 .040;   %        x label
        220 .890 .890 .015 .040;   %        y label
        219 .910 .940 .070 .040;   %        x cursor readout
        217 .910 .890 .070 .040]}; %        y cursor readout

  c = [100 10001 101 10000 206001];  c = [c c]; % trace colors
  S.cfg = inifile('afilt.mat');                 % save configuration data in plt\ini\afilt.mat
  S.ftypes = {'Butter' 'Bessel', 'Cheby1'  'Cheby2' 'Elliptic'}; t10 = ones(1,10);
  S.lbx = '\omega (radians/sec)';  lby = {'dB' 'Phase \circ'};
  band = {'low pass' 'high pass' 'band pass' 'stop band'};  yli = {[-90 60] [-1000 200]};
  s = pltinit(0,t10,'Right',6:10,'HelpFile','*/apps/afilt.htm','-Ytick',[-180 0 180],'moveCB',@mv,...
     'xy',p{1},'DualCur',-5,'TraceID',S.ftypes,'Pos',[925 525],'Ylim',yli,'LabelX',S.lbx,'LabelY',lby,...
     'addtag',{'Save' @cfg},'closeReq',@cfg,'TIDcback',@cb,'TraceC',c,'+Ytick',-140:20:0,'Options','Xlog');
  % switch to solid grid lines with new graphics engine (R2014b or later)
  if getappdata(0,'Mver') >= 8.04   plt xright TGLgrid;  end;
  plt('box',p{2});           % create a frame for the 4 filter parameters
  S.n    = plt('edit',p{3} ,[5 1 25],   @cb,'BackGr',[1 1 1]/12,'label','Ord~er:');
  S.band = plt('pop', p{4} ,band,       @cb,'index',3,'swap');
  S.dec  = plt('pop', p{5} ,1:5,        @cb,'index',3,'label','Decades:','hide');
  S.pts  = plt('pop', p{6} ,50*2.^(1:5),@cb,'index',2,'label','Points:', 'hide');
  v = {[2 .01 9] [40 10 110] [.1 .01 10] [.3 .01 10]};
  t = {'Passband ripple' 'Stopband ripple' 'Cutoff frequency' 'frequency 2'};
  S.sli = plt('slider',p(7:10),v,t,@cb,{'6' '6' '%3.2f 6 2' '%3.2f 6 2'});
  S.etr = text(p{11}{:},'','units','norm','horiz','center','color',[.2 .6 1]);
  text(.19,-.1,VerString,'units','norm','color',[0 .7 1],'fontsize',9);
  % create phase delay button and prototype checkbox
  uic('units','norm','Pos',p(12:13),'String',{'phase delay' 'prototype'},'backgr',[.7 .8 .5],...
      'Style',{'pushbutton' 'checkbox'},'value',1,'Callback',{@delay @cb},'Buttond',{@delay ''});
  uc = [exp((0:.01:2)*pi*1j) NaN -99 99 NaN -99j 99j]; % unit circle and xy axes
  r = pltinit('Fig',gcf,0,t10,uc,'xyAXc',505000,'TRACEid',0,'ENAcur',[t10 0],...
       'TRACEc',[c 303000],'Styles','nnnnnnnnnn-','Markers','oooooxxxxxn',...
       'LabelX','','LabelY','','Options','-All noBtn','xy',p{14});
  set(findobj(gca,'marker','+'),'MarkerSize',20); % easier to see larger cursors
  S.tr = [s; r];  set(gcf,'user',S);              % save handles for callback
  set(findobj(gcf,'marker','x'),'MarkerSize',9);  % increase marker size for poles
  h = plt('slider',S.sli,'get','obj');  h(:,5)=[];  h = h(:);
  set([h; S.etr],'buttond','plt ColorPick;');
  for k = 1:length(h) setappdata(h(k),'m',{'backgr' h}); end;
  if exist(S.cfg) load(S.cfg);  % load configuration file if it exists -------------
                  S = get(gcf,'user');  plt('edit',S.n,'value', cf{1});
                  plt('pop',[S.band S.dec S.pts],'index',num2cell(cf{2}));
                  plt('slider',S.sli,'set',num2cell(cf{3}));  set(h,'background',cf{4});
                  set(S.etr,'color',cf{5});                   set(gcf,'position',cf{6});
  end;
  cb;  cur -1 updateH -1;     % initialize plot and center the cursor
  plt('HelpText','on',htxt);  % add help text
% end function afilt

function cb(a,b) % callback function for all objects
  S = get(gcf,'user');  N = plt('edit',S.n);              % get handle list and filter order
  dec = plt('pop',S.dec);  pts = 50*2.^plt('pop',S.pts);  % get # of decades and # of points
  bi = plt('pop',S.band);                                 % get band index (low,high,band,stop)
  [Rp Rs Wn Wm] = dealv(plt('slider',S.sli));
  if bi>2 Wn = [Wn Wm]; v = 'visON'; else v = 'visOFF'; end;
  plt('slider',S.sli(4),v);                               % frequency 2 visible for type 3,4
  X = round(log10(mean(Wn)) - dec/2 + .5);  X = logspace(X,X+dec,pts); % X-axis (radians/sec)
  af(bi,N,Wn,Rp,Rs,X,S.tr);          % compute response and set trace data
  cur(-1,'xlim',X([1 end]));         % set Xaxis limits for frequency response plot
  cur -1 updateH;                    % update cursor
  h = find(get(S.tr(5),'y') < -Rs);  % compute Elliptic transition ratio
  if     isempty(h)                    h = 0;  % no values of the trace in the stop band
  elseif (bi-2)*(bi-3)                 h = X(h(1))/Wn(1);  % here for low pass or stop band
  else    h = find(diff([h inf])>1);   h = Wn(1)/X(h(1));  % here for high pass or pass band
  end;
  set(S.etr,'string',prin('Elliptic ~, transition ~, ratio: ~, %5v',h));
  t = plt('show'); t = t(find(t<6)); t = [t t+5]; z = t+10; plt('show',[t z 21]);  % for traceID callback
  % find smallest x & y values for all visible traces on the pole/zero plot
  x = -1e-09;  y = x;
  for k = S.tr(z)'  x = min([x get(k,'x')]);  y = min([y get(k,'y')]);  end;
  x = [1.05*x -.2*x];  y = 1.05*y;  y = [y -y]; % x & y limits to insure that we see all the roots
  cur(get(get(S.tr(end),'parent'),'user'),'xylim',[x y]); % set pole/zero plot axis limits
  plt('HelpText','off');
  delete(findobj('type','figure','user',gcf)); % delete any open phase delay or group delay figures
% end function cb

function mv()  % move cursor callback - moving main cursor also moves cursor on pole/zero plot
  if get(findobj(gcf,'string','prototype'),'value') return; end;  % doesn't make sense for prototypes
  cid = getappdata(gcf,'cid');  xy = cur(cid(1),'get');           % get cursor position
  x = real(xy);  if ~x return; end;
  cur(cid(2),'setActive',cur(cid(1),'getActive'));  % same active trace
  cur(cid(2),'updateN',1,0,x); % put the pole/zero plot cursor on the imaginary y axis
%end function mvCur

function cfg(a,b) % write configuration file
  S = get(gcf,'user');  s = plt('slider',S.sli(1),'get','obj');
  cf = {plt('edit',S.n);    plt('pop',[S.band S.dec S.pts]);  plt('slider',S.sli);
        get(s(2),'backgr');  get(S.etr,'color');              get(gcf,'position')};
  num = get(S.tr(1),'user');  den = get(S.tr(2),'user');  S = S.cfg;  save(S);
  if nargin % give visual feedback that the Save tag was clicked
    s = findobj(gcf,'string','Save');  set(s,'vis','off'); pause(.2);  set(s,'vis','on');
    if strcmp(get(gcf,'SelectionT'),'alt')  % right click on save tag also saves afilt.txt
      prin(-strrep(S,'.mat','.txt'),'----- %s -----\nnum:{  %20V}\nden:{  %20V}\n\n',...
         'Butter',num{1},den{1},  'Bessel',  num{2},den{2},  'Cheby1',num{3},den{3}, ...
         'Cheby2',num{4},den{4},  'Elliptic',num{5},den{5});
    end;
  end;
% end function cfg

function delay(a,b)
  g = gcf;  L = get(g,'SelectionT');  L = L(1)~='a';  % L is true for Left click
  S = get(g,'user');  X = get(S.tr(1),'x');  Y = get(S.tr(6:10),'y');  y = [];
  for k=1:5  % loop thru all 5 filter types
    u = -unwrap(Y{k}*pi/180);  du = diff(u);  i = find(abs(du)>1);  du(i) = du(i-1);
    v = u(1) + [0 cumsum(du)];
    if L fig = 'phase delay';  y = [y; v./X];
    else fig = 'group delay';
         y = [y; (v([2:end end]) - v([1 1:end-1])) ./ (X([2:end end]) - X([1 1:end-1]))];
    end;
  end;
  plt(X,y,'Options','Xlog','TraceID',S.ftypes,'LabelX',S.lbx,'LabelY','Seconds','FigName',fig);
  set(gcf,'user',g);

function af(bi,N,Wn,Rp,Rs,X,g)  % compute analog filter response & set trace data ----------------
                                % bi = 1,2,3,4   for low pass, high pass, pass band, stopband 
  Xj = X*j;  od = rem(N,2);  rs = sqrt(10^(.1*Rs)-1);      % stopband ripple
  rp2 = 10^(.1*Rp)-1;  rp = sqrt(rp2);  rp1 = 1/rp;        % passband ripple
  if bi<3 Wn = Wn(1);                                      % low pass or high pass
  else    bw = diff(Wn);  Wn = sqrt(prod(Wn));  q = Wn/bw; % bandpass or stopband
  end;
  proto = get(findobj(gcf,'str','prototype'),'value');     % true if prototype checkbox checked
  for e=1:5  % e  = 1,2,3,4,5 for butterworth, besself, cheby1, cheby2, ellip
    k = 1;  z = [];
    switch e
    case 1, p = exp(j*(pi*(1:2:N-1)/(2*N) + pi/2));                          % butterworth ---------
            p = [p; conj(p)];  p = p(:);  if od p = [p; -1];  end;
    case 2, m = N:-1:0;  d = N-m;
            p = fac(N+d)./(2.^d.*fac(m).*fac(d));                            % besself ---------
            p = roots(p)/p(end)^(1/N);
    case 3, mu = asinh(rp1)/N;  p = exp(j*(pi*(1:2:2*N-1)/(2*N) + pi/2)).';  % cheby1 ---------
            p = sinh(mu)*real(p) + j*cosh(mu)*imag(p);
            if ~od k = sqrt((1+rp2)); end;
    case 4, mu = asinh(rs)/N;  p = exp(j*(pi*(1:2:2*N-1)/(2*N) + pi/2)).';   % cheby2 ---------
            if od z = j ./ cos([1:2:N-2 N+2:2:2*N-1]*pi/(2*N))';  M=N-1;
            else  z = j ./ cos((1:2:2*N-1)*pi/(2*N))';            M=N;
            end;
            t = [1:M/2; M:-1:M/2+1];  z = z(t(:)); % Organize complex pairs
            p = 1 ./ (sinh(mu)*real(p) + j*cosh(mu)*imag(p));
    case 5, k1 = rp/rs;  k1p = 1-k1^2;  p = [];                              % ellip ---------
            if abs(1-k1p) < eps kr = 0;
            else Ne = N*ellipke(k1^2); kr = Ne/ellipke(k1p); % kr = K(k)/K'(k)
            end;
            set(0,'user',kr);  % find elliptic parameter m so that K(m)/K'(m) = kr
            m = fminsearch(inline('abs(ellipke(min(1,max(x,0)))/ellipke(min(1,max(1-x,0))) - get(0,''user''))'),.5);
            if m>0 & m<1
              em = ellipke(m);  jj = 1-od:2:N-1;  % find zeros on the y axis (imaginary)
              [s,c,d] = ellipj(jj*em/N,m*ones(size(jj)));
              z = j ./(sqrt(m)*s(find(abs(s) > eps)));  z = [z conj(z)];
              set(0,'user',k1p);  setappdata(0,'rp1',rp1);  % save k1p and rp1 for inline function
              r = fminsearch(inline('abs(getappdata(0,''rp1'') - tan(asin(ellipj(u,get(0,''user'')))))'),ellipke(1-m));
              [S,C,D] = ellipj(em*r/Ne,1-m);  p = -complex(d*S*C.*c,s*D)./(1-(d*S).^2);  pj = conj(p);
              if od pj = pj(2:end); else k = sqrt((1+rp2)); end; % remove first pj for odd N, adjust gain for even N
              p = [p pj];
            end;
    end; % end switch e
    [a,b,c,d] = zp2ss(z,p,real(prod(-p)/prod(-z))/k);  % Transform to state-space
    m = size(b,2);  [x,y] = size(c);  ey = eye(y);  v = 0*c;  zy = zeros(y);  zym = zeros(y,m);
    switch bi
      case 1, a = a*Wn;                  b = b*Wn;                                                    % low pass
      case 2, d = d-c/a*b;  c = c/a;     b = -Wn*(a\b);           a =  Wn*inv(a);                     % high pass
      case 3, a = Wn*[a/q ey; -ey zy];   b = Wn*[b/q; zym];       c = [c v];                          % band pass
      case 4, d = d-c/a*b; c = [c/a v];  b = -[Wn/q*(a\b); zym];  a = [Wn/q*inv(a) Wn*ey; -Wn*ey zy]; % band stop
    end;
    den = poly(a);  [Z,P,k] = ss2zp(a,b,c,d);                       % caclulate num/den polynomials
    ld = length(den); ln = length(Z)+1;  num = [zeros(1,ld-ln) k*poly(Z)];
    if ~proto  z = roots(num);  p = roots(den);  end;               % recalculate roots for p/z plot
    H = polyval(num,Xj)./polyval(den,Xj); r = [e e+5];              % calculate transfer function
    set(g(r),'x',X,{'y'},{20*log10(abs(H)); angle(H)*180/pi});      % set traces (magnitude & phase)
    set(g(r+10),{'x'},{real(z); real(p)},{'y'},{imag(z); imag(p)}); % set traces (pole/zero)
    NUM{e} = num;  DEN{e} = den;                                    % save for afilt.mat file
  end;  % end for e = 1:5
  set(g(1:2),{'user'},{NUM; DEN});                                  % save for afilt.mat file
% end function af

function n = fac(n)   % Vectorized factorial. Only needed for Matlab 6 since its factorial
  c = cumprod([1 1 2:max(n)]);  n(:) = c(n+1);  % function does not accept vector inputs
