% motion.m (Brownian motion simulation)-------------------------------
%
% After starting this program, click on the Help tag in the menu box near the lower left
% corner of the figure. There you will be able to get more information about how to use
% this application and some of the programming techniques used. You can also see this help
% information by typing "plt help" at the command prompt, then clicking on the "motion"
% link in the "Math & engineering topcs" section.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function motion(steps)           % steps specifies the # of display iterations on startup
  p = {[ .03 .50 .025 .35]; [.40  .01 .21 .034];  % Positions: speed and # of points sliders
       [.005 -.2 .1   .18]; [.015 .94 .06 .04 ]}; % Positions: Arrow size popup & run/walk button
  if ~nargin steps = 20; end;    % If steps not specified, the default is 20 steps
  if ischar(steps) steps=0; end; % to run forever, call with steps=0 or a string argument
  mx = 512;                      % maximum number of markers
  colors  = {'red';'green';'blue';'yellow';'magenta';'none'}; % marker colors
  markers = {'o';'>';'<';'^';'v';'square';'diamond'};         % marker styles
  S.L = pltinit(0,zeros(1,mx),'TraceID',[],'LabelX','','LabelY','','Options','Ticks-X-Y',...
       'Xlim',[-1 1],'Ylim',[-1 1],'TraceC',[.2 .6 1],'HelpFile','*/apps/motion.htm',...
       'XYaxc',[.3 .3 .3],  'markersize',10,...           % fixed  marker size
       'MarkerFaceColor',colors(ceil(6*rand(1,mx))),...   % random marker face colors
       'MarkerEdgeColor',colors(ceil(5*rand(1,mx))),...   % random marker edge colors
       'Marker',         markers(ceil(7*rand(1,mx))),...  % random marker type
        0,0);                                             % velocity arrows
  set(S.L(end),'color',[1 1 1]/5);                        % set velocity arrow color
  S.S = plt('slider',p{1},[60 1 200],'speed','',2);
  S.N = plt('slider',p{2},[128 2 mx 2 mx],'#points','',2);
  S.A = plt('pop',   p{3},{' small ' 'medium' ' large ' ' none '},'LabelY','Arrows');
  S.B = uic('string','Walk','pos',p{4},'callback',{@walk -1});
  set(gcf,'user',S,'tag','motion');
  if ~ishandle(S.B) return; end; % in case it was closed during pause
  % walk(0,0,steps-1)            % start it going
  funcStart({@walk steps-1});    % alternative to above line. Allows immediate return.
% end function motion

function walk(a1,a2,steps)
  S = get(gcf,'user');
  if get(S.B,'user') set(S.B,'user',0,'string','Walk'); return;
  else               set(S.B,'user',1,'string','STOP');
  end;                                     % toggle walk/stop button
  mx = length(S.L)-1;                      % max number of markers
  w = complex(randn(mx,1),randn(mx,1))/3;  % marker velocities - Gaussian distribution
  p = [];  v = [];  l = [];  c = 0;  tic;
  while steps & ishandle(S.B) & get(S.B,'user') % while walking do ...
    steps = steps-1;  c = c+1;                  % count update steps
    n = plt('slider',S.N);                      % get number of markers
    if length(l)~=n  l = S.L(1:n);  w(1:length(v)) = v;  v = w(1:n);
                     set(S.L,'vis','off'); set(l,'vis','on');
                     px = get(l,'x');  py = get(l,'y');  p = complex([px{:}],[py{:}])';
    end;
    p = p + plt('slider',S.S)*v/1000;                       % update positions
    set(l,{'x'},num2cell(real(p)),{'y'},num2cell(imag(p))); % move markers
    a = plt('pop',S.A);                                     % get arrow size
    if a<4 q = Pquiv(p,a*v/4);                              % here if arrows are on
           set(S.L(end),'x',real(q),'y',imag(q),'vis','on');
    else   set(S.L(end),'vis','off');                       % here if arrows are off
    end;
    % Update velocity: Negate Vx if x is out of bounds; Negate Vy if y is out of bounds.
    v = v - 2 * complex(real(v) .* (abs(real(p))>1), imag(v) .* (abs(imag(p))>1));
    if ~mod(c,50)                                  % show update rate every 50 steps
      ups = sprintf('motion: %.2f updates/sec',c/toc);
      if ~strcmp(get(gcf,'tag'),'motion') return; end; % don't update wrong figure
      set(gcf,'name',ups);                             % put it in the title bar
    end;
    drawnow;
  end;  % end while
  if ishandle(S.B) set(S.B,'user',0,'string','Walk'); end;
% end function walk
