% hermite.m ---------------------------------------------------------
%
% Fits a polynomial thru six points on a plane so that the polynomial also matches prescribed
% derivatives at those points (i.e. Hermite interpolation).
%
% Click on the Help tag in the menu box to read more information about this application.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function hermite()
  htxt = { ...
  'Click on one of the white arrowheads, then click' 'on the "Modify" button. Then drag the arrow head' ...
  'and observe that the green trace changes so that' 'it is tangent to all the arrows. Try this again,' ...
  'except click on one of the arrow tails instead.'  .41+.99i 2i ...
  'If it''s combersome to click the Modify button for every edit, check "Persistent edit mode"' ...
  'below. Then it will be in data edit mode as soon as you click on the head or tail of an arrow.' ...
  .015+.11i 2i 'To learn the mathematics behind this interpolation, left click on the "Help" tag in' ...
  'the menu box. Or right click on that tag to toggle this on screen help text on or off.' ...
  .1+.24i 'color' 101};
  x = .5:.8:5;  vt = x + (humps(x/5)/25 + 2)*1j;              % vector field tail coordinates
  vh = ([29 24 26 61 44 36] + [72 50 -91 -18 67 -43]*1j)/100; % head coordinates (tail offset)
  X = 0:.02:5;  Y = herm(vt,tan(angle(vh)),X);
  p  = [1 .15 .11 .84 .87;  -1 .01 .88 .11 .08];              % axis & TraceID box position
  % create the plot ----------------------------------------------------------------------
  pltinit(Pquiv(vt,vh),X,Y,'pos',[800 530],'xy',p,'Xlim',[-.2 5.2],'Ylim',[0 7],...
    'MotionEdit',@motionE,'Linewidth',{1 5},'TraceID', {'vectorField' 'interpolated'},...
    'FigBKc',90909,'TRACEc',[10101 4000],'HelpText',htxt,'HelpFile','*/apps/hermite.htm');
  % create the Modify button and the Persistent edit mode checkbox -----------------------
  uicontrol('pos',[ 35 350  45 20],'string','Modify','callback','plt xleft Yedit 8;');
  uicontrol('pos',[280   8 120 20],'style','checkbox','string','Persistent edit mode',...
            'callback','cur(0,''Emode'',char(79-get(gcbo,''value'')*13));');

function motionE(a)
  plt helptext;  warning off;  hc = a{3};      % hc is the handle of the moving edit cursor
  h = get(hc,'user');  h = h{1};               % h is handle of the line being edited
  xy = complex(get(h,'x'),get(h,'y'));         % get current x & y data for edited trace
  n = length(xy);  if mod(n,7) return; end;    % continue only for quiver traces
  xy(a{9}) = complex(get(hc,'x'),get(hc,'y')); % replace data at current index with marker position
  n7 = 1:7:n;  a1 = xy(n7);                    % a1 = tail positions from quiver data
  a2 = xy(n7+1)-a1;  xy = Pquiv(a1,a2);        % a2 = new head positions, xy = new quiver trace
  v = getappdata(gcf,'Lhandles');  v = v(2);   % find the trace 2 handle
  set(v,'y',herm(a1,tan(angle(a2)),get(v,'x')));  % update trace 2 data
  set(h,'x',real(xy),'y',imag(xy));               % update data of edited trace

function y = herm(xy,m,xx) % Hermite interpolation
  % xy: A vector of points that the polynomial must go thru.
  %     The real part is the x locations and the imaginary part is the y locations.
  % m:  A vector of slopes that the derivative of the polynomial must have at the
  %     x locations specified in the xy input.
  % xx: The resulting hermite polynomial will be evaluated at these x locations.
  % y:  The result of this evaluation (which will be the same size as xx). 
  xy = xy(:);  m = m(:);  x = real(xy); n = length(x);  N = 2*n-1;
  A = (x * ones(1,N+1)) .^ (ones(n,1) * (0:N));
  B = A(:,1:N) .* (ones(n,1) * (1:N));
  p = [A; [zeros(n,1) B]] \ [imag(xy); m]; % coefficients of the hermite polynomial
  y = polyval(flipud(p),xx);
  % Recent versions of Matlab (2016b or later) will implicitly expand arrays where applicable.
  % This allows you to replace the expressions for A & B with these simpler and faster expressions:
  %      A = x .^ (0:N);
  %      B = A(:,1:N) .* (1:N);
  % I didn't use the simpler forms here because I wanted this to run on older Matlab versions.
