% gpsLog.m --- (version 15Aug23) ---------------------------------------------------
%
% Reads the GPS data in the file specified, or in the file "gpsLog.csv" in the
% plt\ini folder if an input argument is not included. If the gpsLog.csv file does
% not exits, the file will be created using simulated GPS data. Two plots are then
% created, an altitude/speed plot and an xy position plot.
%
% Click on the help tag in the menu box to read the help file for this application.
% Reading the help file will allow you to make full use of the program features.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function gpsLog(in1)
  f = inifile('gpsLog.csv');  % GPS log filename
  if nargin
    if length(in1)<2
      [f,p] = uigetfile(f,'Select a universal csv gps data file');  f = [p f];
      if ~exist(f) disp('No csv file was selected'); return; end;
    else s = strfind(in1,filesep);
         if length(s) f = in1;  else s = strfind(f,filesep);  f = [f(1:s(end)) in1]; end;
         if isempty(strfind(f,'.')) f = [f '.csv']; end;
         if ~exist(f) disp(['file ' f ' was not found']); return; end;
    end;
  elseif ~exist(f) % if the log file doesn't exist, create one using simulated data
                   % Define a route used to generate the simulated GPS data -------
    r = [4847 -3376 52; 4849 -3764 42; 4789 -3766 41; 4701 -3857 39; 4665 -3856 39;
         4665 -4151 37; 4300 -4155 36; 4303 -3226 02; 5899 -3229 65; 5900 -3686 75;
         5688 -3740 80; 5590 -3976 98; 5570 -4151 71; 4821 -4153 55];
    r = [r(:,1)/1e5+37.3 r(:,2)/1e5-122 r(:,3)]; % waypoints in Sunnyvale CA
    try rand('state',0); catch; end;             % ignore warnings with newer Matlab versions
    v = [0 0 0];  m=0;  p = r(1,:);                    % start at 1st waypoint with v=0;
    for l = 2:length(r(:,1))                           % do all legs
      e = r(l,:);  steps = round(dst(p,e)/12);         % gps fix approximately every 12 meters
      for n = 1:steps                                  % do all points on this leg
        m=m+1;  lla(m,:) = p;  v = v.*(rand(1,3)+.5);  % Change velocity by random amount
        vr = (e-p)/(steps+1-n);                        % required velocity to get to endpoint
        w = .08+((n<5)+(n>steps-15))*.5;               % weighting for vr
        v = w * vr + (1-w) * v;   p = p + v;           % compute next gps position [lat lon alt]
      end;
    end;
    t = 5e5 / 7;                         % arbitrary start time (seconds since midnight)
    t = t + cumsum(2 + .05*rand(m,1));   % random sampling intervals between 2 & 2.05 sec
    hr = floor(t/3600); t = t - 3600*hr; % hours
    mn = floor(t/60);   t = t - 60*mn;   % minutes
    % Done generating simulated GPS data ---------------------------------------------
    %Universal csv file format: ------------------------------------------------------
    % Column 1: HH:MM:SS.mmm        (HH=hours  MM=minutes  SS=seconds  mmm=miliseconds
    % Column 2: latitude            (degrees)
    % Column 3: longitude           (degrees)
    % Column 4: altitude            (meters)
    prin(-f,'Time,Lat,Lon,Altitude\n');                             % create the csv file header
    prin(+f,'%02d:%02d:%06.3f,%.6f,%.6f,%05.2f\n',[hr mn t lla]');  % write all lines of data
  end;
  loadf(f);

function loadf(f,z) % Read universal csv format (see GPS Babel for the format definition)
  if figpos('width')<1380 | length(getappdata(0,'pltSmall')) % small/big positioning constants
        sA=[560 640]; sB=.63; sC=.64; sD=[580 5 615 725]; sE=.36; sF=.33; sG=11; sH=10; sI=1.26;
  else  sA=[640 640]; sB=.66; sC=.67; sD=[660 5 740 840]; sE=.32; sF=.1;  sG=13; sH=11; sI=1.16;
  end;
  if isempty(strfind(f,'gpsLog.csv'))  htxt = '';  htx = '';  % only use helptext with
  else                                                       % the default input file
    htxt = {...
     'Click the "o" button (lower left corner)' ...
     'to toggle the track markers on/off.' .46+.82i 'color' 6001 'fontsize' sG 2i ...
     'The cursor in this figure is always linked to the cursor' ...
     'in the altitude/speed plot. You can go to a new point' ...
     'of the track by clicking in either figure.' .09+.725i 'fontsize' sG+3 2i ...
     'The Lat/Long/Time readout near the bottom' ...
     'edge of this figure shows the data associated' ...
     'with the current cursor position.' .13+.2i 'fontsize' sG+3 2i ...
     'You can advance smoothly forward or backward along the track by' ...
     'holding down the left or right mouse button after clicking on either' ...
     'Xaxis label i.e. "\bf\itMinutes since\rm" or "\bf\itEast (meters)\rm"' ...
     .09+.595i 'color' 6001 'fontsize' sG 2i ...
     'Check the "Aspect" checkbox to set the aspect ratio to one,' ...
     'showing you the true shape of the track.' .09+.48i 'fontsize' sG};
    htx = {'To choose your preferred speed units'  'click on the purple text (showing ground' ...
           'speed) near the center of the lower'   'edge of the figure. Also click on' ...
           'the green text to choose your'         'preferred altitude units.' ...
           .04+.996i 'color' 5001 'fontsize' sH};
  end;
  if nargin<2 z = fopen(f); end;  % open file if it hasn't been already
  s = fgetl(z);                   % read header line (lat/lon are the only required fields)
  q = [-1 find(s==',')-1 length(s)];  fn = ['gpsLog: ' f];
  Ctime=0; Calt=0; Clat=0; Clon=0; Ctime=0; Ccourse=0; Cspeed=0; Cdate=0;
  for k=1:length(q)-1 name = [s(q(k)+2:q(k+1)) '  ']; 
      switch lower(name(1:3))
         case 'tim',          Ctime   = k;  % time      is found in column Ctime   of the csv
         case {'alt' 'ele'},  Calt    = k;  % altitude  is found in column Calt    of the csv
         case 'lat',          Clat    = k;  % latitude  is found in column Clat    of the csv
         case 'lon',          Clon    = k;  % longitude is found in column Clon    of the csv
         case 'cou',          Ccourse = k;  % course    is found in column Ccourse of the csv
         case 'spe',          Cspeed  = k;  % speed     is found in column Cspeed  of the csv
         case 'dat',          Cdate   = k;  % date      is found in column Cdate   of the csv
       end;
  end;
  hr = []; mn = []; sc = []; lla = []; course = []; speed = []; dt = []; day = 0; HRp = 0;
  fth = 1; ftm = 0; fts = 0;  % fake time (use if time stamps missing)
  while 1
    s = fgetl(z);  if ~ischar(s) break; end;
    q = [-1 find(s==',')-1 length(s)];
    if Ctime  tim = s(q(Ctime)+2:q(Ctime+1));  HR = day + sscanf(tim(1:2),'%d');
              mn  = [mn;  sscanf(tim(4:5),'%d')];   sc  = [sc;  sscanf(tim(7:end),'%f')];
              if HR < HRp day = day + 24;  HR = HR + 24; end;  HRp = HR;  hr  = [hr; HR];
    else hr = [hr; fth];  mn = [mn; ftm];  sc = [sc; fts];  % here if no time stamps
         if fts==59  fts=0;  if ftm==59 fth=fth+1; ftm=0; else ftm=ftm+1; end;
         else        fts=fts+1;   % advance the fake time by 1 second
         end;
    end;
    if Calt alt = s(q(Calt)+2:q(Calt+1)); else alt = '0'; end;
    lla = [lla  sscanf([s(q(Clat)+2:q(Clat+1)) ' ' s(q(Clon)+2:q(Clon+1)) ' ' alt],'%f')];
    if Ccourse course = [course; sscanf(s(q(Ccourse)+2:q(Ccourse+1)),'%f')]; end;
    if Cspeed  speed  = [speed;  sscanf(s(q(Cspeed)+2:q(Cspeed+1)),'%f')]; end;
    if Cdate & isempty(dt) dt = s(q(Cdate)+2:q(Cdate+1)); fn = [fn ' (' dt ')'];end; % save the 1st date
  end;
  fclose(z);  lla = lla';
  t0 = [hr(1) mn(1) floor(sc(1))];           % start time [hours minutes seconds]
  hms = [3600;60;1];  t0s = t0*hms;          % start time (seconds)
  t = ([hr mn sc]*hms - t0s)/60;             % time in minutes since start
  [d c] = dst(lla(2:end,:),lla(1:end-1,:));  % distance/course between successive samples
  v = d ./ (60*diff(t));                     % v = dx/dt. Convert from meters/min to meters/sec
  v = [v(1); v];  c = [c(1); c];  dis = 0;   % make v & c the same length as t
  if ~Cspeed  speed  = 0*v; dis = [0 0 1]; end; % if speed missing from file, plot nothing for that trace
  ele = lla(:,3);  y0 = min(ele);  y1 = max(ele);  dy = (y1-y0)/20;  yl = [y0-dy y1+dy];
  Mver = version;  if Mver(1)=='6' ys = 'prin("%5w (%5w)'; else ys = 'prin("%5w \{\\color{red}(%5w)\}'; end;
  ys = {[ys ' %s",@V1(@IDX),@V2(@IDX),@Q3)'] 'pos' [sI  .4 ] 'buttond' @ystr 'color' 10001 'V2' speed};
  xs = {'prin("%5w %s ",@V3(@IDX),@Q4)'      'pos' [sI 1.25] 'V3' lla(:,3) 'buttond' @xstr 'color' 100};
  xy = [-1 .004 .62 .065 .11;  302 .55 -.06 0 0];  % positions: TraceID box & Xaxis Label
  % create altitude/speed plot --------------------------------------------------------
  L = pltinit(t,[ele v speed c],'Right',[2 3],'Options','-X-Y','Pos',sA,'moveCB',@CurCB,...
    'Xstring',xs,'Ystring',ys,'Subplot',[72 28],'AXIScb',@AXcb,'addTag',{'Climb' @climb},'FigName',fn,...
    'HelpFile','*/apps/gpsLog.htm','TraceID',{'Alt' 'sCalc' 'sFile'},'FigBKc',1020,'DIStrace',dis,...
    'HelpText',htx,'xlim',[0 t(end)],'ylim',yl,'Xlabel',prin('Minutes since %d:%02d:%02d Z       ',t0),...
    'xy',xy,'Ylabel',{{'Elevation' '(meters)'} 'True Course' 'm/s'},'cTrace',[100 10001 13100 101]);
  if getappdata(0,'Mver')<7 set(get(gca,'ylabel'),'units','norm','pos',[-.11 .5]); end; % fix bug in Matlab 6
  gc = gcf;  cidA = get(gca,'user');   % get cursorID
  setappdata(gc,'Q3','m/s');  setappdata(gc,'Q4','meters');
  set(L(2),'z',v);  set(L(4),'z',unwrap(c*pi/180)*180/pi);
  ax = getappdata(gcf,'axis');  set(ax(2),'ylim',[-20 380]);
  if Ccourse  axes(ax(2));  lc = line(t,course,'color',[1 .3 0]);
              axes(plt('box',[.93 .82 .05 .06],0));  u = [L(4) lc];
              text(.01,.8,'Calc','color',[0  1 1],'fontw','bold','buttond',@swap,'user',u);
              text(.01,.3,'File','color',[1 .3 0],'fontw','bold','buttond',@swap,'user',u);
  end;
  set([findobj(gcf,'marker','+'); findobj(gcf,'marker','o')],'MarkerSize',16);
  p = {[.910 .046 .057 .620];  [.910 .018 .065 .343];  % positions: solid & dashed track width popus
       [.942 .033 .050 .023];  [.942 .006 .050 .023];  % positions: solid & dashed color selection frames
       [.012 .945 .060 .025];  [ sC  .009 .040 .042];  % positions: Avg edit object & kmz button
       [ sB   0   .370 .060]};                         % position:  Box around km related objects
  plt('box',p{7},0);                                   % Create a box around the kmz related objects
  h = uic('Pos',p(3:4),'style','frame','Backgr',{10101 10100},'user',lla,...
          'enable','inactive','buttond','plt ColorPick;');
  c = prin('9{%d ~, }9{%de ~, }0',1:9,1:9);
  h = [plt('pop',p{1},c,      'index',2,'label',' Solid track width:','Offset',0,'hide',h);
       plt('pop',p{2},[1:9 0],'index',6,'label','Dashed track width:','Offset',0,'hide',h); h];
  uic('pos',p{6},'str','kmz','user',h,'enable','inactive','buttond',@kmz);
  plt('edit',p{5},[6 0 50],@avg,'incr',2,'label',{'Avg' [.06 .023i]},'tag','avg');  % smoothing for speed/course
  avg;  ax = getappdata(gcf,'axis');  axes(ax(2));  plt xleft Yedit 3;        % enable xView slider on top axis
  Cpolar = 40007863;  Cequat = 40075017;                % polar & equatorial circumference (meters)
  N = lla(:,1); E = lla(:,2);                           % convert Lat/Long to North/East data
  E = (E - min(E)) * cos(mean(N)*pi/180) * Cequat/360;  % Zero will be the x position of the western most fix
  N = (N - min(N)) * Cpolar/360;                        % Zero will be the y position of the southern most fix
  xl = [min(E) max(E)];    xl = xl + [-.04 .04]*max(1,diff(xl));  % limits (East/West)
  yl = [min(N) max(N)];    yl = yl + [-.04 .04]*max(1,diff(yl));  % limits (North/South)
  p = { sD;                [.125 .115 .865 .875];                 % positions: plot, axis
       [.277 .008 sE .04]; [.275 .058 .085 .023];                 % positions: box, Aspect button
       [.01  .88 .05 .023]};                                     % positions: load button
  xs = {'0' 'Color' [1 .7 0] 'User' [lla 60*t+t0s] 'Tag' 'LLT'};  % for Lat/Lon/Time
  ys = 'prin(" Sample ~,  no. %d",@IDX)';  if Mver(1)=='6' zs = ''; else zs = '\\color{green}'; end;
  zs = {['prin("Path from start =' zs ' %5w %s",@V4(@IDX),@Q1)'] 'pos' [-sF 1.55] 'buttond' @xstr};
  % create position plot --------------------------------------------------------
  l = pltinit(E,N,'Pos',p{1},'xy',p{2},'Options','-Y','Xlim',xl,'Ylim',yl,'moveCB',@CurCB,'Styles','o',...
      'addTag',{'TAS' @TAS},'Xstring',xs,'Ystring',ys,'Zstring',zs,'HelpFile','*/apps/gpsLog.htm',...
      'Figname',fn,'Link',gc,'LabelX','East (meters)','LabelY','North (meters)','HelpText',htxt);
  setappdata(gcf,'Q1','meters');  setappdata(gcf,'V4',[0; cumsum(d)]);  % save path length from start
  plt('box',p{3});                                                      % box around Lat/Lon & Time readouts
  uic('pos',p{4},'str','Aspect','ena','inactive','buttond',@aspect,'user',{gca [get(gca,'xlim') get(gca,'ylim')]});
  uic('pos',p{5},'str','Load',  'ena','inactive','buttond',@loadb,'user',f);
  uic('style','checkbox','pos',p{4}+[-.11 0 .015 0],'str','allPoints','callback',...
  'h=getappdata(gcf,"Lhandles"); z=get(h,"z"); set(h,"z",getappdata(h,"z")); setappdata(h,"z",z);');
  set(findobj(gcf,'marker','+'),'MarkerSize',32);                      % easier to see larger cursors
  cidB = get(gca,'user');  set(gcf,'user',cidA);  set(gc,'user',cidB); % save cursor IDs in figure userdata
  set(l,'z',0*get(l,'x'));  CurCB(0);

function loadb(a,b) % load button
   [f,p] = uigetfile(get(gcbo,'user'),'Select a universal csv gps data file');  f = [p f];  z = fopen(f);
   if z<0  disp('No csv file was selected'); return;
   else  if ~strcmp(get(gcf,'SelectionT'),'alt')  close(get(gcbo,'parent')); end;
         loadf(f,z);
   end;

function swap(a,b)  % swap data and colors between the two traces on the upper subplot
  u = get(gcbo,'user');  r = u(1);  s = u(2);
  y = get(r,'y');  c = get(r,'color');
  set(r,'y',get(s,'y'),'color',get(s,'color'));
  set(s,'y',y,'color',c);

function xstr(a,b)  % Xstring callback
  u = getappdata(gcf,'Q4');       v = 'Q4';  f = isempty(u);
  if f u = getappdata(gcf,'Q1');  v = 'Q1'; end;
  switch u
    case 'meters',   u = 'km';      m = .001;      % convert from meters to kilometers
    case 'km',       u = 'feet';    m = 1e7/3048;  % convert from kilometers to feet
    case 'feet',     u = 'Kfeet';   m = .001;      % convert from feet to thousands of feet
    case 'Kfeet',    u = 'miles';   m = 1/5.28;    % convert from thousands of feet to miles
    case 'miles',    u = 'meters';  m = 1609.344;  % convert from miles to meters
  end;
  ax = getappdata(gcf,'axis');  a = ax(1);  cid = get(a,'user');
  if f  setappdata(gcf,'V4',getappdata(gcf,'V4')*m); setappdata(gcf,'Q1',u);
  else  if u(1)=='K' v = '    (thousands of feet)'; else v = ['(' u ')']; end;
        setappdata(gcf,'Q4',u);  set(get(a,'ylabel'),'string',{'Elevation' v});
        L = getappdata(gcf,'Lhandles');  y = get(L(1),'y')*m;  set(L(1),'y',y);  setappdata(gcf,'V3',y);
        y0 = min(y);  y1 = max(y);  dy = (y1-y0)/20;  cur(cid,'ylim',[y0-dy y1+dy]);
  end;
  cur(cid);

function ystr(a,b)  % Ystring callback
  switch  getappdata(gcf,'Q3');
    case 'm/s',    u = 'km/hr';  m = 3.6;            % convert from m/s to km/hr
    case 'km/hr',  u = 'knots';  m = 1000/1852;      % convert from km/hr to knots
    case 'knots',  u = 'mph';    m = 1852/1609.344;  % convert from knots to mph
    case 'mph',    u = 'ft/s';   m = 5280/3600;      % convert from mph to ft/s
    case 'ft/s',   u = 'm/s';    m = 0.3048;         % convert from ft/s to m/s
  end;
  setappdata(gcf,'Q3',u);  ax = getappdata(gcf,'axis');  ar = ax(3);  set(get(ar,'ylabel'),'string',u);
  L = getappdata(gcf,'Lhandles');
  y = get(L(3),'y')*m;  setappdata(gcf,'V2',y);  set(L(3),'y',y);
  y = get(L(2),'y')*m;  setappdata(gcf,'V1',y);  set(L(2),'y',y,'z',get(L(2),'z')*m); 
  y0 = min(y);  y1 = max(y);  dy = (y1-y0)/20;  set(ar,'ylim',[y0-dy y1+dy]);
  cur(get(ax(1),'user'));

function climb(a,b)
  q4 = getappdata(gcf,'Q4');  q3 = getappdata(gcf,'Q3');     % get units for altitude and speed
  u = (6+strfind('meters km     feet   Kfeet  miles',q4))/7;  
  m = [1 1000 .3048 304.8 1609.344];  m = m(u(1));           % convert altitude to meters
  u = (5+strfind('m/s   km/hr knots mph   ft/s',q3))/6;
  q = [60 50/3 1852/60 160.9344/6 18.288];  q = q(u);        % convert speed to meters/min
  L = getappdata(gcf,'Lhandles');    ax = get(L(1),'parent');  xlbl = get(ax,'Xlabel');
  t = get(L(1),'x');  e = get(L(1),'y');  v = get(L(2),'y');       % get time/altitude/speed
  fil = sscanf(get(findobj(gcf,'tag','avg'),'str'),'%d');          % moving average filter length
  roc = diff(e) ./ diff(t);  roc = filter(ones(1,fil)/fil,1,roc);  % rate of climb
  t = t(2:end);  a = atan((roc*m) ./ (v(2:end)*q)) * 180/pi;       % angle of climb
  y1 = roc(fil+2:end);  y0 = min(y1);  y1 = max(y1);  dy = (y1-y0)/20;  yl = [y0-dy y1+dy];
  Y1 =   a(fil+4:end);  Y0 = min(Y1);  Y1 = max(Y1);  dY = (Y1-Y0)/20;  Yl = [Y0-dY Y1+dY];
  plt(t,roc,t,a,'pos',[660 5 740 690],'Xlabel',get(xlbl,'string'),'Ylim',yl,'FigBKc',20,...
    'FigName',get(gcf,'name'),'Subplot',[50 50],'Xlim',get(ax,'xlim'),'link',gcf,...
    'Ylabel',{['climb rate (' q4 '/minute)'] 'climb angle (degrees)'},'xy',[-2 .01 .47 .06 .24]);
  AX = getappdata(gcf,'axis');  cur(get(AX(2),'user'),'ylim',Yl);

function avg
  L = getappdata(gcf,'Lhandles');  Avg = findobj(gcf,'str','Avg');
  n = plt('edit',get(Avg,'user')); n2 = round(n/2);  n = 2*n2;  % get length of moving average
  v = get(L(2),'z');  c = get(L(4),'z');
  if n  b = ones(1,n)/n;  v = filter(b,1,v);           c = filter(b,1,c);
        e = ones(1,n2) ;  v = [v(n2+1:end) v(end)*e];  c = [c(n2+1:end) c(end)*e];
  end;
  set(L(2),'y',v);  set(L(4),'y',mod(c,360));
  setappdata(gcf,'V1',v);    ax = getappdata(gcf,'axis');
  y0 = min(v);  y1 = max(v);  dy = (y1-y0)/20;  set(ax(3),'ylim',[y0-dy y1+dy]);

function AXcb
  cid = get(gcf,'user');  if isempty(cid) return; end;
  obj = cur(cid,'obj');  g = get(obj(1),'parent');                % get other figure
  L = getappdata(g,'Lhandles');  l = getappdata(gcf,'Lhandles');  % get line handles
  l = l(1);  t = get(l,'x');  xl = get(get(l,'parent'),'Xlim');   % get time values and main axis limits
  z = zeros(size(t));  a=z;  a(find(t<xl(1) | t>xl(2))) = NaN;    % set points outside limits to NaN
  if ~get(findobj(g,'string','allPoints'),'value'); b=a; a=z; z=b; end;
  set(L,'z',z);  setappdata(L,'z',a);

function aspect(a,b)
  plt helptext off;  u = get(gcbo,'user');  n = length(u);  ax = u{1};  cid = get(ax,'user');
  if strcmp(get(gcf,'SelectionT'),'alt')  % here for right click (undo aspect limits)
        cur(cid,'xylim',u{n});  if n>2 set(gcbo,'user',u(1:n-1)); end;
  else  p = get(ax,'pos').*get(gca,'pos');
        xl = get(ax,'xlim');  yl = get(ax,'ylim');  set(gcbo,'user',[u {[xl yl]}]);
        xp = p(3)/diff(xl);  yp = p(4)/diff(yl);  % pixel width & height (in meters)
        if xp>yp  dx = diff(xl)*(xp/yp - 1);  xl = xl + [-dx dx]/2;  cur(cid,'xlim',xl);
        else      dy = diff(yl)*(yp/xp - 1);  yl = yl + [-dy dy]/2;  cur(cid,'ylim',yl);
        end;
  end;

function CurCB(a)                    % cursor move callback
  cid = get(gcf,'user');             % Don't do anything when CurCB is called the
  if isempty(cid) return; end;       % first time (before the cid's are saved)
  if ~nargin plt helptext; end;      % turn off help text
  [xy k] = cur(-1,'get');            % get the cursor position
  cur(cid,'updateN',k);              % link the cursors
  h = findobj(gcf,'tag','LLT');      % find the Xstring (Lat/Long/Time display)
  if isempty(h) h = cur(cid,'obj');  % get the cursor objects from the other figure
                h = findobj(get(h(7),'parent'),'tag','LLT'); % find LLT in the other figure
  end;
  u = get(h,'user'); % get the lla/time data and display it on the multiline Xstring
  t = u(k,4); hr = floor(t/3600); t = t - 3600*hr; % hours
              mn = floor(t/60);   t = t - 60*mn;   % minutes
  set(h,'str',prin('  Lat/Long:  %f   %f ~,        Time:  %d:%02d:%06.3f Z',u(k,1:2),hr,mn,t));

function kmz(a,b)                                      % kmz button callback ----------
  set(gcbo,'visible','off');                           % give visual feedback that .kml was saved
  c = get(gcf,'name'); c = [c(9:strfind(c,'.csv')) 'kml']; f = fopen(c,'wt'); % .kml in same folder as .csv
  h = get(gcbo,'user');  lla = get(h(3),'user');       % get GPS data [lat lon alt]
  cid = get(gcf,'user');  if isempty(cid) return; end;
  obj = cur(cid,'obj');  g = get(obj(1),'parent');     % get other figure
  z = get(getappdata(g,'Lhandles'),'z');                       % get z data
  lla = transpose(lla(find(z==0),[2 1 3]));  n = size(lla,2);  % n is the # of GPS fixes visible in position plot
  ws = plt('pop',h(1));  wd = plt('pop',h(2));                 % get solid & dashed track widths
  if ws>9 ex = '<extrude>1</extrude> '; ws = ws-9; else ex = ''; end; % extrude option for solid track
  s1 = '</coordinates> </LineString>\n</Placemark>';
  s3 = '<Style id="%s"> <LineStyle> <color>ff%02x%02x%02x</color> <width>%d</width> </LineStyle> </Style>\n';
  s2 = sprintf(s3,'solid',round(255*fliplr(get(h(3),'backgr'))),ws);
  s3 = sprintf(s3, 'dash',round(255*fliplr(get(h(4),'backgr'))),wd);
  s5 = '<Placemark> <styleUrl>#%s</styleUrl> <LineString> %s<altitudeMode>absolute</altitudeMode> <coordinates>\n';
  s4 = sprintf(s5,'solid',ex);  s5 = sprintf(s5,'dash','');  s6 = '  %.6f,%.6f,%.2f\n';
  fprintf(f,['<kml xmlns="http://www.opengis.net/kml/2.2">\n<Document>\n' s2 s3]);
  if ws<10                fprintf(f,s4); fprintf(f,s6,lla);          fprintf(f,s1);      end; % skip if solid width = 0
  if wd<10  for k = 2:2:n fprintf(f,s5); fprintf(f,s6,lla(:,k-1:k)); fprintf(f,s1); end; end; % skip if dash width = 0
  fprintf(f,'</Document> </kml>\n'); fclose(f);
  if ~strcmp(get(gcf,'SelectionT'),'alt') & exist('zip')  % here for left click (create kmz file)
     z = strrep(c,'.kml','');  zip(z,c); movefile([z '.zip'],[z '.kmz']);  % compress kml file to kmz
     if exist('z') delete(c); end;                                         % delete kml file
  end;
  pause(.3); set(gcbo,'visible','on');  % pause to give visual confirmation that the file was written.

function TAS(a,b)
  h = cur(get(gcf,'user'),'obj');  % get the cursor objects from the other figure
  g = get(h(7),'parent');          % g is altitude/speed figure
  L = getappdata(g,'Lhandles');  rc = strcmp(get(gcf,'SelectionT'),'alt');  % true if right click
  if rc  v=L(3);  c = [1 .3 0];  fi = '\rightarrowf'; % use speed/course from file
  else   v=L(2);  c = [0  1 1];  fi = '';             % use calculated speed/course
  end;
  l = findobj(g,'type','line','color',c);           % find line containing course data
  if isempty(l) disp('no speed/course data found'); return; end;
  c = get(l(1),'y');  v = get(v,'y');  u = getappdata(g,'Q3');      % get course, speed, speed units
  [n h] = cur(get(g,'user'),'getActive');                           % h is handle of position trace
  i = find(get(h,'z')==0);  n = length(i);                          % restrict to visible points
  radian = pi/180;  v = transpose(v(i));  c = radian*transpose(c(i));
  B = [v.*cos(c) v.*sin(c) ones(n,1)]\v.^2;  Wx=B(1)/2;  Wy=B(2)/2; % backslach linear regression
  TAS = sqrt(B(3)+Wx^2+Wy^2);  [Wdir, Wspeed] = cart2pol(-Wx,-Wy);  % wind reported as "from"
  t = '  TAS = %5w %s ~,   Wind at %5w %s from %3d\\circ 7{ }%s';
  set(findobj(gcf,'tag','LLT'),'string',...
     prin(t,TAS, u, Wspeed, u, round(mod(Wdir/radian,360)), fi)); % put results in lat/lon/time box

function [m, d] = dst(p1,p2) % Great circle distance & course (meters/degrees).
  % p1/p2 = [lat lon]. This method is accurate along lines of longitude or latitude,
  % and otherwise accurate only over short distances (up to a few hundred miles)
  Cpolar=40007863; Cequat=40075017;  % WGS84 polar/equatorial circumference (meters)
  lat1 = p1(:,1)/360;  lon1 = p1(:,2)/360;  lat2 = p2(:,1)/360;  lon2 = p2(:,2)/360;
  [d m] = cart2pol( (lon1-lon2)*Cequat.*cos((lat1+lat2)*pi), (lat1-lat2)*Cpolar);
  if nargout>1 d = mod(90-d*180/pi,360); end; % convert course to degrees
