% gauss.m ----------------------------------------------------------------
%
% This example script plots the sums of several uniform random variables.
% * Shows the advantage of passing plot data in cell arrays when the traces contain
%   different number of data points.
% * Shows how the line zData can be used to save an alternative data set which in this
%   example is the error terms instead of the actual correlations. A checkbox allows
%   you to tell the plot to show the alternative data set. The label for the checkbox is
%   rotated 90 degrees so that it can fit in the small space to the left of the plot.
% * Note the use of the 'FigName' and 'TraceID' arguments.
% * Note the appearance of the greek letter in the x-axis label.
% * The 'Options' argument enables the x-axis cursor slider (which appears just below the
%   peak/valley finder buttons), adds the Print tag to the menu box, and lastly removes
%   the LinX/LogX and LinY/LogY selectors from the menu box.
% * Uses the 'DIStrace' argument so that gauss.m starts off with some traces disabled.
% * Shows an example of the use of the 'MotionZoom' parameter. To see what it does, create
%   a zoom box by holding both mouse buttons down and draging the mouse in the plot window.
% * The zoom window plot also demonstrates an easy way to copy the trace data from one
%   plot to another (in this case from the main plot to the zoom plot).
% * The first trace is displayed using markers only to distinguish the true Gaussian curve.
% * Demonstrates the use of the 'HelpText' parameter to initialize a GUI with user help
%   information that is cleared when the user begins to use the application. In this case
%   the MoveCB parameter is used to cause the help text to be removed when you click on
%   the plot. The help text is also removed if you click on the checkbox. If you want the
%   help text to reappear, simply right click on the help tag in the MenuBox.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function gauss
  dis = 1:10 > 4;    % initially show just the first 4 traces
  sz = 100;          % size of each uniform distribution
  u = ones(1,sz);    % uniform distribution
  y = u;             % y will be composite distribution
  xx={}; yy={};      % xx/yy holds the data for all the convolutions (2-10)
  dy={};             % dy holds the errors between the convolutions & the gaussian
  s = 1/sqrt(2*pi);  % scale factor for sigma = 1
  mxN = 2:10;        % sum from 2 to 10 uniform distributions
  for n = mxN
    y = conv(y,u);   % convolve with next uniform distribution
    m = length(y);  mean = (m+1)/2;  y = s*y/max(y);
    sigma = sz * sqrt(n/12);
    x = ((1:m) - mean) / sigma;              % change units to sigma (zero mean)
    xx = [xx {x}];  yy = [yy {y}];           % append data for the next convolution
    g = s*exp(-(x.^2)/2); dy = [dy; {y-g}];  % append the data for the next error trace
  end;
  xx = [{x} xx];  yy = [{g} yy];  dy = [{x+NaN}; dy];   % append the 1st trace (true gaussian)
  ct = [1  1  1;  1  0 1; 0  1  1;  1  0  0; .2 .6  1;  % trace colors
        1 .6 .2; .2 .3 1; 1 .2 .6; .5 .5 .5;  0  1  0];
  htxt = {'Try opening a zoom box by' 'dragging the mouse while' 'holding down the shift key.' ...
          '' 'Observe that a new'  'window appears' 'showing the contents' 'of the zoom box.' .03+.98i };
  L = pltinit(xx,yy,'FigName','Sum of uniform distributions','DIStrace',dis,'xlim',[-3.7 3.7],...
      'ylim',[-.03 .43],'LabelX','Standard deviation (\sigma)','LabelY','','TRACEc',ct,...
      'pos',[0 inf 650 600 58],'TraceID',prin('Gauss ~, {Sum%2d!row}',mxN),'MotionZoom',@lens,...
      'MoveCB','plt helptext;','Options','-X -Y','HelpText',htxt,'HelpFile','*/apps/gauss.htm');
  set(L(1),'Linestyle','--');     % dashed line to distinguish the true gaussian trace
  set(L,{'user'},dy);             % save error terms in the user data
  uic('style','CheckBox','pos',[.03 .40 .03 .04],'backgr',get(gcf,'Color'),'callback',@chkBox);
  uic('pos',[.015 .33 .06 .04],'string','Cum','callback',@cum);
  text(-.12,.4,'Plot errors only','units','norm','Rotation',90,'color','white'); % checkbox label
% end function gauss

function cum(a,b)                                            % Cumulative distribution buttton
  L = getappdata(gcf,'Lhandles');                            % get line & axis handles
  x = get(L,'x');  y = get(L,'y');  vis = get(L,'vis');  ct = get(L,'color');
  if max(y{2}) < .1 y = get(L,'user'); end;
  c = {};  for k = 1:10  c = [c; {cumtrapz(x{k},y{k})}]; end;
  h = plt(x,c,'xlim',[-4 4],'ylim',[-.05 1.1],'TraceID',cur(0,'TraceIDs'),'link',gcf,...
    'TRACEc',ct,'DIStrace',strcmp(get(L,'vis'),'off'),'FigName','Cumulative distribution function');
  set(h(1),'Linestyle','--');

function chkBox(a,b)                                                      % checkbox callback function
  L = getappdata(gcf,'Lhandles');   a = get(L(1),'parent');               % get line & axis handles
  yl = {[-.03 .430] [-.04 .033]};  set(a,'ylim',yl{get(gcbo,'value')+1}); % set y axis limits
  y = get(L,'y');  set(L,{'y'},get(L,'user'),{'user'},y);                 % swap the y & user data
  plt('grid',a);  plt helptext;  % update grid lines and turn help text off
% end function chkBox

function lens(xyLim)                                         % MotionZoom function
  zoomFig = 'Gauss magnifying lens'; g = gcf;  f = findobj('name',zoomFig);
  if isempty(f) % create the zoom window if it doesn't already exist
     h = getappdata(gcf,'Lhandles'); % get x/y vectors for all 10 traces into 1 cell array
     vis = get(h,'vis');  ct = get(h,'color');  xy = [get(h,'x') get(h,'y')]';
     p = get(gcf,'pos');                              % position of main figure
     p = p + [p(3)+18 0 0 0];  p(3) = 445i;           % position of zoom figure
     L = plt(xy{:},'TraceID',cur(0,'TraceIDs'),'LabelX','','LabelY','',...
      'DIStrace',strcmp(get(h,'vis'),'off'),'FigName',zoomFig,'Pos',p,'link',g,...
      'TRACEc',ct,'Options','-All','ENApre',0,...
      'xy',[1 .223 .097 .748 .87; -1 .004 .630 .123 .34]);
     set(L(1),'Linestyle','--');  f = gcf;  ax = gca;
     tid = findobj(f,'user','TraceID');  a = axes;  % create new axis for area display
     pr = {'pos' 'xlim' 'ylim' 'color' 'xcolor' 'ycolor' 'XtickLabel' 'YtickLabel' 'TickLen'};
     p = get(tid,pr);  p{1}(2) = .26;  set(a,pr,p);  t = [];
     for k = 1:10  t = [t text(.07,.1-k,'','color',ct{k})]; end;  setappdata(ax,'t',t);
  end;
  a = getappdata(f,'axis');  L = getappdata(f,'Lhandles');  x = get(L,'x');  y = get(L,'y');
  cur(get(a,'user'),'xylim',xyLim);  v = {};  x1 = xyLim(1);  x2 = xyLim(2);
  for k=1:10  xk=x{k}; yk=y{k};  j = find(xk>x1 & xk<x2);                % find portion inside lens
              if length(j)>1 vv = trapz(xk(j),yk(j)); else vv = 0; end;  % compute area of that portion
              v = [v; {prin('%7V',vv)}];
  end;
  set(getappdata(a,'t'),{'string'},v);  % display area under the curve in the box below the TraceID box
  figure(g);                            % restore focus to main window
% end function mZoom  
