% circles12.m ---------------------------------------------------------
%
% Creates 4 figures, showing 5 different solutions to this problem:
%
%      *****************************************************
%      *****************************************************
%      **                                                 **
%      **            THE 12 CIRCLE PROBLEM:               **
%      **    Draw 12 circles in a plane so that every     **
%      **    circle is tangent to exactly 5 others.       **
%      **                                                 **
%      *****************************************************
%      *****************************************************
%
% A button and an edit pseudo object are added to one of the figures (solution 1) which lets
% you rotate the image and control the rotation speed. Circles12 will normally start with
% solution 1 rotating slowly (with a rotation speed setting of 2). Click on the left/right
% side of the speed setting number to decrease/increase this number respectively. (Higher
% numbers rotate faster.) The number of display updates per second is displayed underneath
% the plot to give you a measure of the system's plotting speed. Note that the funcStart
% function is used so that Circles12 returns to the command prompt even while the plot is
% continuing to rotate. If you call Circles12 with an argument (of any type or value),
% then the plot won't start rotating until you press the "start rotation" button.
% The rotation of this plot doesn't add anything to the puzzle, but this feature was
% included merely to introduce the concepts required to create a moving plot.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function circles12(in)
  N = 600;                        % number of points per circle
  cN = exp((1:N)*pi*2i/(N-1))';   % N point unit circle
  gg = [1 0 0; 1 0 1; 0 .66 0; .2 .3 1];  % red, purple, green, and blue circles
  % Solutions #1 and #2.  Solution #1 is the solution that was published with the puzzle
  % when I ran across it. You are not likely to see any of the other solutions published,
  % since I invented them as well as formulas for the circle sizes and positions.
  ID = prin('middle ~, outer ~, {group%d!row}',[1 1 1 1 1 2 2 2 2 2]);
  g  = gg([1 2 ceil((11:20)/5)],:);            % 4 colors: inner, outer, group1, group2
  pi5 = pi/5;  s = csc(pi5);  u5 = ones(1,5);
  c5 = exp((0:4)*pi5*2i);                      % 5 point unit circle
  rg1 = 1/(s-1);  t = 1 + 2*rg1;               % radius of group 1 circles
  rg2 = roots([s^2-1, -2*(s*sqrt(t)+rg1), t]); % radius of group 2 circles
  b = c5*s*rg1*exp(pi5*1i);  fg = [];
  for k = 1:2                                  % plot solution 1 & 2
    R = rg2(k);  c = [0 0 b c5*s*R];           % c = center location of all 12 circles
    r = [1 R*(s+1) rg1*u5 R*u5];               % r = radii of all 12 circles
    h = plt(cN*r + repmat(c,N,1),'LabelY','','LabelX','','TraceID',ID,'Options','Ticks',...
       'Pos',[1250-620*k 9 600 0],'LineWidth',3,'HelpFile','*/apps/circles12.htm',...
       'FigName',sprintf('12 Circles (solution %d)',k),'TraceC',g,'Link',fg);
    if k==1                                          % here for the first solution only
      fg = gcf;                                      % link all the figures to the first
      cur(0,'xylim',13.5*[-1 1 -1 1]);               % modify axis limits for solution 1
      up = text(-2.47,-15.28,'','color',[.3 .7 1]);  % updates per second (solution 1 only)
      rs = plt('edit',[258 15 24 16],[2 0 9],'label','Rotation sp~eed:'); % rotation speed
      if getappdata(0,'Mver') < 7  s1 = 'rotate';  s2 = 'stop';
      else s1 = '<html> &nbsp; start<br>rotation</html>'; % strings for muli-line button
           s2 = '<html> &nbsp; stop<br>rotation</html>';  % (not supported in Matlab 6)
      end;
      bt = uicontrol('str',s1,'pos',[5 240 58 40],'callback',@rt,...  % start/stop button
           'tag','circ12','fontweight','bold','user',{rs h cN c r up s1 s2});
    end;
  end;

  % Solution #3 ------------------------------------------------------
  % It's amazing that there is even one solutions to this puzzle, but to my eye this
  % solution is both the most surprising and the most beautiful. Looking at the tangent
  % point at the bottom where four circles meet, the radii of the red, purple, green,
  % and blue circles are 17/11, 2.2, 3, 4 respectively. Looking at the tangent point at
  % the far right or far left where four circles meet, the radii of the red, purple, green,
  % and blue are 1, 2, 3, 4 respectively. The only requirement for that sequence is that
  % the middle two numbers have the same sum as the end two numbers, so for example
  % 1,3,4,6 would work equally well although it might not be as pleasing to the eye.
  % For the 1,2,3,4 sequence chosen here, notice that there are only 6 different circle sizes.
  % All 3 blue circles are the same size and all 3 green circles are also the same size.
  % Yet the red circles come in two sizes as do the purple circles. Surprising indeed!
  % As a bonus, this solution looks somewhat like a face - of an alien or perhaps a bug?
  g  = gg(1+mod(0:11,4),:);       % 3 circles of each color
  v = 1:4;  r = (12+5*v)./(12-v); % radii of vertical set of 4
  c = [v-1 9-v complex(4,r-7)];   % center location of all 12 circles
  r = [v v r];                    % radii of all 12 circles
  pltinit(cN*r + repmat(c,N,1),'FigName','12 Circles (solution 3)',...
      'Pos',[10 650 640 669],'Options','Ticks','Xlim',[-1.6  9.5],'Ylim',[-7.5 4.8],...
      'LabelY','','LabelX','','LineWidth',3,'TraceID',prin('{circle%d!row}',1:12),...
      'TraceC',g,'Link',fg,'HelpFile','*/apps/circles12.htm');

  % Solutions #4 and #5  ------------------------------------------------------
  % These two solutions are plotted in a single axis, so 24 circles in total.
  % However there are still only 12 traces. (Each trace is used to draw 2 circles and an
  % NaN is inserted between them so a line isn't drawn from the end of the first circle
  % to the beginning of the next.) Solution 4 looks pretty similar to solution 3 except
  % that only 4 circle sizes are used instead of 6 and also solution 4 is symetric over
  % 120 degree rotations whereas solution 3 has only left/right symmetry.
  % Solution 5 gives us a way to make the original problem more difficult. In addition
  % to the requirement that each of the 12 circles must be tangent to 5 others, we can
  % add the requirement that no more than 3 different circle sizes are allowed. With that
  % requirement, solutions 1 thru 4 don't work, but solution 5 hits the mark.
  % If you can derive a formula to compute the exact size of the mid-sized circle
  % (2.30930734142) you win the mathematition of the week award!
  u = exp(pi/6i); a = u^4;  w = u*a;  w = [u w w*a];  s3 = sqrt(3);  v = [-1 1 -1i*s3];
  rr = [1 2;  % radii of the two smallest circle sizes  (arbitrary) - solution 4
        1 5]; % radii of the smallest & largest circles (arbitrary) - solution 5
  ss = [3.7865066985;   % solution 4 triangle size (gives circles of sizes 1,2,3,4.11684397)
        2.18890105932]; % solution 5 triangle size (gives circles of sizes 1,5,2.30930734142)
  g  = gg(ceil((1:12)/3),:);  % 3 circles of each color
  for k = 1:2                 % compute solutions 4 and 5
    e = rr(k,:);  s = ss(k);  t = v*s;  rs=[]; c=[]; % rs saves radii. c saves circle centers
    for r = e    a = s*s3 + r;   q = .5 * (a^2 + s^2 - r^2) / (r+a);
                 rs = [rs r r r q q q];   c  = [c t t+(q-r)*w]; 
                 t =  t + diff(e)*w;  s = diff(t);  s = s(1)/2;
    end;
    z{k} = cN*rs + repmat(c,N,1); % z{1} is solution 4.  z{2} is solution 5
  end;
  plt([z{1}; NaN+ones(1,12); .82*z{2} + 10.5-1.8i],'Pos',[125 650 1060 515],'LineWidth',3,...
      'FigName','12 Circles (solutions 4 & 5)','Xlim',[-5.7 16],'Ylim',[-8 3],'TraceC',g,...
      'HelpFile','*/apps/circles12.htm','LabelY','','LabelX','','Options','Ticks','Link',fg);

  figure(fg);                     % put the figure for solution 1 on top
  if ~nargin funcStart(@rt); end; % start rotation if circles12 called with no arguments

  % There are actually two more solutions to this puzzle that I have not plotted in any of
  % the figures. Notice that for both solutions 1 and 2, the red circle is tangent to the
  % inside of the five green circles. In each case you can make the red circle bigger so
  % that it is tangent to the outside of those same five green circles. I will call those
  % solutions 1a and 2a.
 
function r = rt(a,b) % rotation button callback -----------------------------------------
  p = findobj('tag','circ12');  h = get(p,'user');  % get pushbutton handle and user data
  if length(strfind(get(p,'str'),'stop')) set(p,'str',h{7}); return; end; % clicked to stop motion
  set(p,'str',h{8});  l = h{2};  cN = h{3};  c = h{4};  r = h{5};  so = -1;  pause(.001);
  while ishandle(p) & length(strfind(get(p,'str'),'stop'))       % clicked to start motion
     s = plt('edit',h{1});                  % get the rotation speed
     if s ~= so  m = 0;  so = s;  tic; end; % reset counter when speed control changes
     c = c.*exp(s^3/5000i);  m=m+1;         % rotate the circle centers
     if ~mod(m,99) set(h{6},'str',sprintf('%.2f updates/sec',m/toc));  end;
     for k=3:12  v = c(k)+cN*r(k);  set(l(k),'x',real(v),'y',imag(v)); end;
     drawnow;
  end;
% end function rt
