% carlo.m (Monte Carlo simulation) -----------------------------------
%
% Monte Carlo simulation of the Sam Loyd's Carnival Dice game.
% The rules of this game are as follows:
% -  You bet 1 dollar to play. You then roll 3 dice
% -  If one six appears you get 2 dollars
% -  If two sixes appear, you get 3 dollars
% -  If three sixes appear, you get 4 dollars
% -  Otherwise, you get nothing 
% Is this a good bet to make?
%
% Three traces are created:
%     - left hand axis:  accumulated winnings
%     - right hand axis: earnings per bet
%     - right hand axis: expected earnings per bet
%
% The first two traces are displayed as they are computed, i.e. every
% time the die are rolled, a new value is appended to the trace and
% the plot is updated so you can watch the function grow in real time.
%
% A second axis is added near the top of the figure to show the 3 die.
% For each die, a line with dots as markers is added for each of the
% six faces, with only one of these lines being visible at a time.
% A square patch is also added for each die for the visual effect.
%
% There are several ways you can start the program:
% carlo                     - No bets occur until you click a button.
% carlo(n) or carlo n       - Makes n bets, then stops
% carlo(0) or carlo 0 or
% carlo('run') or carlo run - Makes bets continuously until you click stop.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

function carlo(in1)
  p = {[ 0  .135 .095 .760 .760; -1 .005 .89 .16 .1;   % positions: plot & TraceID
        206 .760 .006 .086 .045;  313 -.16 .55 0 0;    % positions: Y edit box & Cursor ID tag
        209 .735 .007 .020 .041];                      % positions: Y edit box label
       [.01 .75 .055 .038];      [.01 .70 .055 .038];  % positions: 1 bet & Continuous buttons
       [.01 .65 .055 .038];      [.21 .945 .360]};     % positions: Clear button, speed slider
  tid = {'Accumulated' 'Earnings/bet' 'expected value'};  c = [10001; 100; 999999];
  lby = {'Accumulated earnings' 'Earnings per bet'};
  S.tr = pltinit(0,[0 0 0],'FigName','Monte Carlo simulation of 3 dice problem',...
    'xy',p{1},'Xstring','char(32)','Right',[2 3],'TraceC',c,'Options','-Y-X Ticks',...
    'LabelX','# of bets','HelpFile','*/apps/carlo.htm','LabelY',lby,'TraceID',tid,...
    'Styles','--:','+Ycolor',c(1),'<color',c(1),'-Ycolor',c(2),'>color',c(2));
  S.ax = getappdata(gcf,'axis');  c = get(gcf,'color');
  plt('box',[.6 .88 .38 .1],c,c,'xlim',[0 5]); % invisible box to contain the 3 die
  patch([0 1.5 3; 1 2.5 4; 1 2.5 4; 0 1.5 3],[zeros(2,3); ones(2,3)],[0 .2 .3]);
  S.tx = text(4.4,.4,' ','color',[0 .7 1],'fontsize',28); % earnings for this bet
  cx = .5;  cy = .5;  e = .3;
  e2 = [-e e];   e3 = [-e 0 e];  e4x = [e2 -e2];    e4y = [-e e2 e];
  e5x = [e4x 0]; e5y = [e4y 0];  e6x = [-e e4y e];  e6y = [e3 e3];
  for k = 1:3  dx = {cx cx+e2 cx+e3 cx+e4x cx+e5x cx+e6x};  % draw all possible rolls
               dy = {cy cy+e2 cy+e3 cy+e4y cy+e5y cy+e6y};  % for each of the 3 die
               S.d(k,:) = [line(dx{1},dy{1}); line(dx{2},dy{2}); line(dx{3},dy{3});
               line(dx{4},dy{4}); line(dx{5},dy{5}); line(dx{6},dy{6})];
               cx = cx + 1.5;
  end;
  set(S.d,'vis','off','linestyle','none','marker','s','markersize',7,...
          'markeredge','none','markerface',[1 1 .5]);
  [S.bet1 S.betC S.clr] = uic('Pos',p(2:4),'String',{'1 bet'; 'Run'; 'Clear'},...
                              'Callback',{{@btn, 0}; {@btn, 1}; {@btn,-1}});
  S.sli = plt('slider',p{5},[900 -100 900],'speed','','','4 7 3');  % speed slider
  S.ups = findobj(gcf,'tag','xstr'); set(S.ups,'color',[.5 .5 1]);  % updates/second
  n = 6^3;  w = -n;                       % cost of placing all possible bets
  w = w + 2*75;                           % 75 ways to win 2 dollars
  w = w + 3*15;                           % 15 ways to win 3 dollars
  w = w + 4*1;                            % 1  way  to win 4 dollars
  set(S.tr(3),'x',[0 1e9],'y',[1 1]*w/n); % set earnings/bet expected value (trace3)
  set(gcf,'user',S);  btn(0,0,-1);        % initialize axis limits and decimation factor
  if nargin
    if ischar(in1) in1 = str2num(in1); end;
    if isempty(in1) | ~in1               % for carlo(0) or carlo('run')
          % btn(0,0,1);                  % click the continuous button
          funcStart({@btn 1});           % alternative to above line (allow immediate return)
    else  for k = 1:in1 btn(0,0,0); end; % otherwise click the 1-bet button in1 times
    end;
  end;
% end function carlo

function btn(a1,a2,a3) % button callbacks -------------------------------------------
  S = get(gcf,'user');
  t = get(S.betC,'string');
  set(S.betC,'string','Run');                  % stop if it was running
  if a3 == -1  S.f = 1;  set(gcf,'user',S);    % CLEAR BUTTON: reset decimation factor
               plt('metricp',S.ax(1),'-both'); %               reset metric prefixes
               set(S.tr(1:2),'x',0,'y',0);     % reinitialize the data and axis limits
               set(S.ax,'xlim',[0 100],{'ylim'},{[-10 10]; [-.1 .1]});  return;
  end;
  if t(1)=='S' return; end;                % If it was running (continuous) we are done
  [xm ym] = plt('metricp',S.ax(1),'get');
  x  = get(S.tr(1),'x')*xm; xe = x(end);   % single or continuous betting comes here
  y  = get(S.tr(1),'y')*ym; ye = y(end);
  yr = get(S.tr(2),'y');
  set(S.betC,'string','Stop');  tic;  m = 20;  f = S.f;
  while ishandle(S.betC)& strcmp(get(S.betC,'string'),'Stop')
    dly = plt('slider',S.sli,'get');        % get slider value - negative for added delay
    n = max(1,dly);                         % - positive for multiple bets per display update
    if ~a3 set(S.betC,'string','Run'); n = 1;  end; % do just one bet
    r = ceil(rand(3,n)*6);        % roll the 3 die n times
    r6 = sum(r == 6);             % count the number of sixes in each column
    for k = 1:n
      e = r6(k) - ~r6(k);         % earnings for this bet
      xe = xe + 1;  ye = ye + e;  % advance x-axis by 1 bet; accumulated winnings in ye
      if ~rem(xe,f)               % if f = 10, append only every 10th point (for example)
        x = [x xe];  y = [y ye];  % add total earnings to line 1 on the left hand axis
        yr = [yr ye/xe];          % add earnings/bet to line 2 on the right hand axis
      end;
    end;
    set(S.d,'vis','off');   r = r(:,end);                   % turn all faces off, select last roll
    set([S.d(1,r(1)) S.d(2,r(2)) S.d(3,r(3))],'vis','on');  % turn on rolled faces
    set(S.tx,'string',num2str(e));                          % show the earnings of the last roll
    while length(x)>6000                                    % plotting more than 600 points is
      f = 10*f;        yr = yr(20:10:end);                  % a waste of time, so increase the
      x = x(20:10:end);  y = y(20:10:end);                  % decimation by a factor of 10
    end;
    set(S.tr(1:2),'x',x/xm,{'y'},{y/ym; yr});               % update traces 1,2
    ya = min(yr);  yb = max(yr);  dy = (yb-ya)/50;          % trace 2 Yaxis limits
    set(S.ax,'xlim',[0 xe/xm],{'ylim'},{[min(y) max(y)]/ym; [ya-dy yb+dy]});
    [xm ym] = plt('metricp',S.ax(1),'both');                % update x & y axis metric prefixes
    m = m-1;  if ~m m=20; set(S.ups,'str',prin('%5w bets/sec',xe/toc)); end;
    if dly<0 pause(-dly/100); else drawnow; end;            % honor speed slider
  end;
  S.f = f;  set(gcf,'user',S);                              % save current decimation factor
% end function btn
