% airspeed.m (Aircraft performance, the power required curve, and non-linear regression)
%
% Click on the Help tag in the menu box to learn you to use this application.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org
% ----- Version: ---- 28Jul23

function airspeed()
  b = 0:4:68;  alt = 0:4:32;    % altitudes (thousands of feet);
  traceID = prin('sea level{ ~, %2d,000 ft}',alt(2:end));  H = '*/apps/airspeed.htm';
  cTRACE  = [0  1  0;  1  0  1;  1  1  0;  0  1  1;  1  0  0;  .2 .6  1;
             1  1  1;  1 .6 .2; .2 .3  1];    cTRACE = [cTRACE; cTRACE];
  htxt = ...  % help text
   {'Pr  =  c \sigma v^3  +  k  /  \sigma v'        .6+.342i 'fontsize' 17  2i ...
    'where:'                                        .47+.26i 'color' [.6 .6 1] 2i ...
    'Pr = power required (for level flight)'        .57+.26i 2i ...
    'c = parasitic drag coef'                       .57+.21i 2i ...
    'k = induced drag coef'                         .57+.16i 2i ...
    '\sigma = air density ratio (\rho / \rho_{SL})' .57+.11i 2i ...
    'v = true airspeed'                             .57+.06i};
  zs = {'prin("Lift portion: %4w%%",@V1(2301*(@LNUM-1)+@IDX))' ...
        'pos' [.08 23.7] 'color' 808000 'buttond',@lcb};        % in @cb: length(tas)=2301
  Tcb = 't=plt("show"); t=t(find(t<=9)); plt("show",[t t+9]);'; % traceID callback
  % Create efficiency and range chart ----------------------------------------------
  Le = pltinit(0,b,'FigName','Efficiency & Range (standard temperature)','HelpFile',H,...
    'Xstring','prin("%5w gph",@Q1*@XVAL)','Ystring','prin(" %5w nm",@Q2*@YVAL)',...
    'LabelX','Percent power','LabelY','nm per gallon','AxisCB',@axcb,'FigBKc',1020,...
    'Pos',[0 0 770 515],'xy',[1 .13 .11 .78 .8; -1 .005 .62 .086 .34],'EnaCur',b<33,...
    'cTRACE',cTRACE,'TraceID',traceID,'TIDcback',Tcb,'Options','-X-Y','Zstring',zs);
  % Display Fuel flow as an alternative x-axis units above the graph
  % Display Range as an alternative y-axis units to the right of the graph
  re = text(0,1.08,'','units','norm','color',[.8 .8 0],'buttondown',@resid);
  ax1 = gca;  hfig = gcf;  ac = get(get(ax1,'ylabel'),'color');  cid1 = get(ax1,'User');
  axt = axes('tag','ts','XaxisLoc','top','YaxisLoc','right','pos',get(ax1,'pos'),...
             'xcolor',ac,'ycolor',ac);
  set(get(axt,'ylabel'),'string','Range (nm)');  axes(ax1); % put primary axis on top

  % Create true air speed chart -------------------------------------------------
  La = pltinit(0,b,'FigName','True Airspeed Chart (standard temperature)','HelpFile',H,...
    'Pos',[785 535 735 548],'xy',[1 .14 .093 .834 .777; -1 .005 .605 .086 .34],...
    'TraceID',traceID,'Link',gcf,'Options','-Y','Xstring','prin("%5w gph",@Q1*@XVAL)',...
    'TIDcback',Tcb,'LabelX','Percent power','LabelY','True Airspeed (knots)        ',...
    'AxisCB',@axcb,'EnaCur',b<33,'cTRACE',cTRACE,'Xlim',[50 85],'HelpText',htxt);
  set([Le(10:end); La(10:end)],'LineStyle','none','marker','o','markersize',8);
  % Display Fuel flow  as an alternative x-axis units above the graph
  ax2 = gca;  cid2 = get(ax2,'User');
  axT = axes('tag','ts','XaxisLoc','top','pos',get(ax2,'pos'),...
             'xcolor',ac,'ycolor',get(gcf,'color'));
  set([get(axt,'xlabel') get(axT,'xlabel')],'string',[ones(1,20)*' ' 'Fuel flow (gph)']);
  axes(ax2);  % put primary axis on top

  % Cursor movement in 1st plot also moves cursor in 2nd plot and visa versa
  s = 'cur(%d,"setActive",@LNUM,-@IDX);'; % negative IDX prevents
  cur(cid1,'moveCB',sprintf(s,cid2));     % infinite recursion
  cur(cid2,'moveCB',sprintf(s,cid1));
  % create a uicontrol and 5 pseudo edit objects in the true airspeed figure
  u = uic('style','text','buttondown',@readCFG,'str',' ','pos',[.535 .955 .455 .033],...
          'fontsize',10,'enable','inactive','backgr',130707,'foregr',606080);
  p = [.915 .1 .033];  q = p+[.04 0 0];  p = {[.25 q] [.25 p] [.42 q] [.42 p] [.89 p]};
  h = plt('edit',p,0,@cb,'incr',{-1 -1 -1 5 4},'format',{8 5 6 6 6},'label',...
      {'parasitic dra~g coef ' 'induced drag ~coef ' 'BSF~C' 'hpM~ax ' 'Usable fuel ~(gal) '})';
  set(gcf,'user',[num2cell(h) {Le La cid1 cid2 hfig gcf u re}]); % save for callbacks
  readCFG;                                                    % initalize from config file
% end function airspeed

function readCFG(a,b)  % read aircraft configuration file from plt\ini folder
  S = get(gcf,'user');  cfg = inifile('airspeed.txt');
  if nargin & strcmp(get(gcf,'SelectionT'),'alt')
    [f,p] = uigetfile(cfg,'Select an aircraft configuration file');  cfg = [p f];
    if ~ischar(cfg) cfg = ''; end;
  end;
  f = fopen(cfg);
  if f<0 disp('aircraft configuration file not found'); return; end;
  c = {};  d = [];
  while length(c)<4 % read four parameters
    t=fgetl(f);  s=strfind(t,': ');  if isempty(s) continue; end;
    t = t(s+2:end);  if isempty(c) c{1} = t;  else c = [c {sscanf(t,'%f')}]; end;
  end;
  fgetl(f); fgetl(f); fgetl(f); % read and discard header lines, then read all data points
  while 1  t = fgetl(f);  if ~ischar(t) break; end;  d = [d sscanf(t,'%f')];  end;
  fclose(f); alt = d(1,:);  tas = d(2,:);  gph = d(3,:);  % alt/tas/gph from file
  sig = (1-alt/145418).^4.2551;                           % density ratio
  [drag,re] = fminsearch(@se,[5e-5 4200],[],alt,tas,gph,c{2},sig,1);
  plt('edit',S(1:5),'val',[num2cell(drag) c(2:4)]);  cb;  % update parameters & plot
  set(S{12},'str',c{1}); % show aircraft name from 1st line of config file
  set(S{13},'str',prin('Regression error: %6w',re/length(tas)),...  % show regression error
            'user',[d; se(drag,alt,tas,gph,c{2},sig,0)]);  % save data and residuals
  
function e = se(in,alt,tas,gph,BSFC,sig,f)  % compute squared error for fminsearch
  Pdrag = in(1);  Idrag = in(2);            % drag coefs passed as inputs from fminsearch
  PR = Pdrag * tas.^3 .* sig + Idrag./(tas .* sig); % power required
  GPH = PR * BSFC / 6.01;                           % convert to gal/hour
  e = GPH-gph;  if f e=norm(e); end;        % if called from fminsearch, return norm error

function cb(a,b) % callback for pseudo edit objects
  S = get(gcf,'user');  Le = S{6};  La = S{7};
  [Pdrag Idrag BSFC MaxHP fuel] = dealv(plt('edit',S(1:5)));
  alt = 0:4:32;  nt = length(alt);           % altitudes (thousands of feet);
  dens = 6.01;                               % density of avgas in pounds/gallon
  sigma = (1-alt/145.418).^4.2551 ;          % density ratio
  ymin=20; ymax=250; tas = (ymin:.1:ymax)';  % airspeed range
  PRi = Idrag./(tas * sigma);                % induced drag term of the PR equation
  PR  = Pdrag * tas.^3 * sigma + PRi;        % power required equation
  liftF = PRi ./ PR;                         % fraction of power used to create lift
  setappdata(gcf,'V1',100*liftF(:));         % save lift percentage as a single column
  p2g = BSFC / dens;  gph = PR * p2g;        % convert horse power to fuel flow
  mpg = repmat(tas,1,length(alt)) ./ gph;    % compute efficency (tas/gph)
  PR = PR * (100/MaxHP);                     % PR is now percent power
  for k = 1:nt
    x = PR(:,k);   i = find(x<=100);  x = x(i);  t = tas(i);  m = mpg(i,k);
    [minx j] = min(x);   xf = x(j:end);      % xf is the front side of the power curve
    [q n] = min(abs(xf - 100*sigma(k)));     % find limit for normally aspirated engine
    Ye = m(n+j-1);   Ya = t(n+j-1);
    if q<.5  X = xf(n);  else  X = NaN; end; % marker invisible if above service ceiling
    set(Le(k+nt),'x',X,'y',Ye);  set(La(k+nt),'x',X,'y',Ya); % normally aspirated limit
    if k==1  pmin = minx-3;  mxm = max(m);  elseif k==6  vmax = max(t);  end;
    set(Le(k),'x',x,'y',m);        set(La(k),'x',x,'y',t);
  end;
  cur(S{8},'xylim',[pmin 100 mxm/2 mxm*1.02]);  % set plot 1 xy limits
  cur(S{9},'xylim',[50 85 .57*vmax vmax]);      % set plot 2 y limits
  p2f = p2g*MaxHP/100;             % conversion factor (% power to fuel flow)
  setappdata(gcf,'Q1',p2f);  setappdata(gcf,'Q2',fuel);
  figure(S{10}); axcb;  figure(S{11});  axcb;  % update top & right axis ticks
  s = get(S{12},'str');  if s(end) ~= ')' set(S{12},'str',[s ' (modified)']); end;

function axcb() % update tick marks on alternate axes (top and right)
  p2f = plt('misc',21); fuel = plt('misc',22); axr = findobj(gcf,'tag','ts');
  if isempty(p2f) return; end;
  set(axr,'xlim',get(gca,'xlim'),'xticklabel',prin('{%4w!col}',get(gca,'xtick')*p2f));
  if isempty(fuel) return; end;
  set(axr,'ylim',get(gca,'ylim'),'yticklabel',prin('{%4w!col}',get(gca,'ytick')*fuel));

function resid(a,b)  % plot residuals from drag coefficient regression
  d = get(gcbo,'user');  p = get(gcf,'pos');  p = [8+sum(p([1 3])) p(2)-5 733 450];
  f = {'fontname' 'Lucida Console' 'fontsize' 9};  res = d(4,:);  n = length(res);
  t = [{'altitude  tas   gph   resid' .64+.98i 'color' 101} f {2i}];
  for k = 1:n  t = [t {prin('%+6W %+7W %+4W  %+7W',d(:,k))}];  end;
  plt(res,'pos',p,'Styles',':','Marker','o','MarkerSize',9,'xlim',[.6 n*1.6],'FigBKc',90000,...
     'Xlabel','Measurement number','Ylabel',' ','link',gcf,'HelpText',[t {.645+.945i} f]);

function lcb(a,b)                                                % lift portion callback
  p = get(gcf,'pos');  p = [8+sum(p([1 3])) p(2)-5 733 450];     % position for new figure
  g = findobj('type','figure','tag',get(gcf,'tag'));  g = g(2);  % find true airspeed chart
  S = get(g,'user');  [n h] = cur(S{9},'getActive');             % get active line
  y = get(h,'x');  x = get(h,'y');  [mp mpi] = min(y);  sz = length(x);
  fn = sprintf('Power required at %d,000 ft',4*n-4);  yl = getappdata(g,'V1')';
  k = 2301*(n-1);  yl = y .* yl(k+1:k+sz)/100;  yp = y-yl;       % in @cb: length(tas)=2301
  tid = {'total' 'pDrag' 'iDrag'};
  plt(x,[y; yp; yl],'pos',p,'Xlabel','True Airspeed (knots','Ylabel','Percent power',...
      'TraceID',tid,'xlim',[x(round(mpi/4)) x(end)],'FigBKc',25,'link',gcf,'FigName',fn);
