% xChart.m --------------------------------------------------------------------------
%
% Data was collected in the "t" array below measuring the performance of 13 different
% computational and graphics tasks, on an AMD Ryzen9 Windows system with 4 different
% Matlab versions. All these tasks are from the plt\demo folder except cbench
% (code specified in help text). Because plt runs in all versions of Matlab since 2001,
% these demo programs offer a good way to compare graphics performance between Matlab
% versions. To achieve more consistent results, Windows was rebooted before each version
% of Matlab was tested.
%
% The rows of the "t" array that represent time are inverted so that longer bars
% always represent better performance. Then each row of the "t" array is normalized
% so that the poorest performing version is given a value of one.
%
% I wouldn't normally overlay so much text over the plot, but in this instance I wanted
% the chart to be self explanatory. If you don't like the colors chosen for any of the
% text items, click on them and adjust the color with the ColorPick window.
% (This is possible because pltpub was called with the 'Cpick' argument.)

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org

%     6.1      2013b    2020a     2021b  <--- Matlab version
%    ------   -------  -------   -------
t = [11.103    27.533   54.266    10.714 ; % demoplt go:  seconds
     13.86     30.734   60.110    12.701 ; % ptime:       seconds
     19.21     8.94     5.94      6.837  ; % julia:       msec per iteration
     1.190     1.156    1.662     1.641  ; % curves:      seconds (cycle, delay=0, best of 5)
     100.9     124.32   73.71     67.8   ; % motion:      updates/sec (10 points)
     15.85     10.18    6.06      4.93   ; % motion:      updates/sec (512 points)
     90.1      119.5    177       178    ; % circles12    updates/sec
     103       80.5     111       137    ; % movbar(1)    updates/sec
     13.22     20.44    16.12     16.5   ; % pltmap(1)    updates/sec (first report)
     19.33     59.6     95        96     ; % wfalltst     updates/sec 
     968       339      192       179    ; % fseries(1)   updates/sec
     9         60       79        78     ; % wfall(1)     updates/sec
     22.525    6.138    2.27      2.05]  ; % cbench       seconds

tst  = ['  demoplt';'   ptime ';'   julia ';'   curves';'   motion';'motion512';'circles12';
        '   movbar';'   pltmap';' wfalltst';'  fseries';'   wfall ';'   cbench'];
ylbl = 'Relative performance (longer bars indicate better performance)';
p    =   {[.050 .045 .940 .94]; [.270 .180 .070 .13];  % plot and TraceID positioning
          [.245 .568 .615 .12]; [.186 .333 .540 .04]}; % text box positioning
clr  = [   1;      4000;   7575;    850000];           % trace colors
tid  = {'ver6.1'  '2013b' '2020a'  '2021b'};           % trace labels
n = size(t,1); k = [1:4 n]; t(k,:) = 1./t(k,:);        % invert seconds (so longer bar is better)
for k = 1:n  t(k,:) = t(k,:) / min(t(k,:)); end;       % for all tests, assign a value of 1 to the slowest version

m = length(tid);  x = m+4;  x = reshape(1:(x*n),x,n);  x = x(1:m,:)'; % x axis spacing
x1=.01; x2=.113; dy=.028i; y1=.995i; y2=y1-dy; y3=y2-dy;              % help text positioning
y4=y3-.12i; y5=y4-dy; y6=y5-dy; y7=y6-dy; y8=y7-dy;
y9=y8-dy; Y0=y9-dy; Y1=Y0-dy; Y2=Y1-dy; Y3=Y2-dy; Y4=Y3-dy;
htxt = {...                                                           % help text
 'These 13 tests were run using 4 different Matlab versions on a Ryzen9 12-core Windows 10 system (32gb ram & Nvidia rtx2060 graphics):' x1+y1 'color' 0 2i ...
 '\bf1.) demoplt:'   x1+y2 'color' 10000 2i ...
 'Seconds to run all 37 demo programs using "demoplt go" (data from the results.txt file)' x2+y2 'color' 0 2i ...
 '\bf2.) ptime:'     x1+y3 'color' 10000 2i ...
 'Same as above but includes the time it takes to start and exit Matlab. Define this startup.m\it\bf:' ...
 '        t0 = clock;  demoplt go;  pause(1); while length(getappdata(0,''demoplt'')) pause(.01); end;' ...
 '        prin(-''c:\\results.txt'',''Elapsed time = %f\\n'',etime(clock,t0)); exit;\rm' ...
 'Then open a dos window in \\matlab\\bin\\win64 and type: "ptime Matlab.exe" (The ptime.exe utility measures execution time.)' x2+y3 'color' 0 2i ...
 '\bf3.) julia:'     x1+y4 'color' 10000 2i ...
 'Milliseconds for each Mandelbrot iteration for the complete image' x2+y4 'color' 0 2i ...
 '\bf4.) curves:'    x1+y5 'color' 10000 2i ...
 'Seconds for a cycle (set delay = 0 and press the cycle button)' x2+y5 'color' 0 2i ...
 '\bf5.) motion:'    x1+y6 'color' 10000 2i 'Updates/sec (10 points selected)' x2+y6 'color' 0 2i ...
 '\bf6.) motion512:' x1+y7 'color' 10000 2i 'Updates/sec (same program as above but with 512 points selected)' x2+y7 'color' 0 2i ...
 '\bf7.) circles12:' x1+y8 'color' 10000 2i 'Updates/sec' x2+y8 'color' 0 2i ...
 '\bf8.) movbar:'    x1+y9 'color' 10000 2i 'Updates/sec' x2+y9 'color' 0 2i ...
 '\bf9.) pltmap:'    x1+Y0 'color' 10000 2i 'Updates/sec' x2+Y0 'color' 0 2i ...
 '\bf10.) wfalltst:' x1+Y1 'color' 10000 2i 'Updates/sec' x2+Y1 'color' 0 2i ...
 '\bf11.) fseries:'  x1+Y2 'color' 10000 2i 'Updates/sec' x2+Y2 'color' 0 2i ...
 '\bf12.) wfall:'    x1+Y3 'color' 10000 2i 'Updates/sec' x2+Y3 'color' 0 2i ...
 '\bf13.) cbench:'   x1+Y4 'color' 10000 2i ...
 'Seconds to run the following script. Depends only on the 4 functions ode45, fft, inv and rand.' ...
 'Different results are likely when choosing different computations:\it\bf' ...
 '                   clear all;  tic;  randn(''state'',0);  y = 0;' ...
 '                   [a,b] = ode45(@vdp1,[0 30000],[1 0]);' ...
 '                   for k=1:16  y = y + fft(randn(1,2\^22)); end;' ...
 '                   inv(rand(2500,2500));  toc\rm' x2+Y4 'color' 0 2i ...
 'The slowest result for each test is assigned a relative performance of 1.0' .15+.345i 'color' 6500 'fontsize' 14 2i ...
 '\itNotes:\rm This chart reveals some dramatic differences between Matlab versions. I''ve noticed that Matlab''s computation speed has'  ...
 'been steadily improving over the years which is confirmed by these tests (see cbench & julia). On the other hand, the speed of' ...
 'creating graphical objects has been declining since the old Ver 6.1 from 2001. But this has reversed dramatically with the latest' ...
 '2021b as evidenced by the first test in this list (demoplt) which gives the best indication of overall speed of creating graphic'    ...
 'objects. Ver 6.1 still wins handily when it comes to the raw speed of updating moving line objects (see fseries and motion512).' ...
 .213+.68i 'color' 1 'fontsize' 9};
pltpub('Cpick',Pvbar(x,0,t),'Linewidth',8,'Pos',[1150 750 25],'xlim',[-1 x(end)+2],...  % create plot
   'HelpText',htxt,'xy',[0 p{1}; -1 p{2}],'TraceID',tid,'TRACEc',clr,'+Ytick',1:11,'+Xtick',x(:,2),...
   '"FontSize',18,'+<FontSize',12,'Ylim',[0 11.3],'LabelX','','LabelY',ylbl,'+XtickLabel',tst);
set(findobj(gcf,'user','TraceID'),'box','on','xcolor','black','ycolor','black');
try set(gca,'XTickLabelRotation',0); end;  % Recent Matlab versions will rotate the xtick lables (unwanted)
plt('box',p(3:4),'none',{1 5000}); % two frames around text (blue and dark green borders respectiveily)
uistack(gca,'bottom');                                  % make frames visible 
g = findobj(gca,'user','grid');  s = get(g,'x');        % get x coordinates of grid line locations
s(43:end) = NaN;                 set(g,'x',s);          % remove vertical grid lines
