% movbar.m -------------------------------------------------------------------
%
% movbar plots a series of 40 random bars and displays a horizontal threshold
% line which you can move by sliding the mouse along the vertical green line.
% As you move the threshold line, a string below the plot reports the number
% of bars that exceed the threshold (demonstrating the use of the plt xstring
% parameter.)
%
% These two buttons are created ---
% Rand: Sets the bar heights to a new random data set.
% Walk: Clicking this once starts a random walk process of the bar heights.
%       Clicking a second time halts this process.
%       The Walk button user data holds the walk/halt status (1/0 respectively)
%       demonstrating a simple way to start and stop a moving plot.
%
% Note that you can move the threshold or press the Rand button while it is
% walking. Also if you click on one of the vertical purple bars, the horizontal
% threshold bar will then follow the upper end of that vertical bar. If movbar
% is called with an input agrument, the value of the argument is ignored, but
% movbar will start as if the walk button has been hit. The walking is initiated
% using the funcStart function so that control returns to the command prompt
% even though movbar continues to update the plot.

% ----- Author: ----- Paul Mennen
% ----- Email:  ----- paul@mennen.org


function movbar(in1);
  htxt = {'\leftarrow click & drag on this green bar to change the horizontal threshold' .035+.99i;};
  u = ones(1,1050);
  S.tr = pltinit(-u,cumsum(u),vdata,'xlim',[-2 42],...   % Line 1 is vertical slider (green)
      'LineWidth',{4 6},'ylim',[0,1070],...              % Line 2 is the random bars (purple)
      'HelpText',htxt,'xstring',{@xstr 'fontsize' 16});  % enable xstring parameter
  S.hz  = line([0 41],[0 0],'color',[0 1 0]);       % create horizontal threshold line
  p = {[10 445 40 22] [10 415 40 22]};  cb = {{@btn,1} {@btn,0}};
  [S.rnd S.wlk] = uic('Pos',p,'Str',{'Rand' 'Walk'},'Callback',cb); % Rand & Walk buttons
  S.ups = text(-.16,.73,'','units','norm','color',[.5 .6 1]);
  text(-.16,.70,'updates/sec','units','norm','color',[.5 .6 1]);
  set(gcf,'user',S);                   % save for callbacks
  cur(0,'update',550);                 % initialize the horizontal level at 550
  % if nargin btn(0,0,0); end;         % walk if input argument provided
  if nargin funcStart({@btn 0}); end;  % alternative to above. (Allows immediate return)
% end function movbar

function t = xstr()  % xstring function: move threshold bar
  S = get(gcf,'user');  if ~isfield(S,'hz') t = [];  return; end;
  y = imag(cur(-1,'get'));  set(S.hz,'y',[y y]);
  t = prin('%d bars > %3W',sum(get(S.tr(2),'y') > y),y); % count bars above threshold
% end function xstr

function btn(h,arg2,r)  % rand and walk button callbacks
  S = get(gcf,'user');
  if r walk(S.tr(2),0);
  else if findstr('S',get(S.wlk,'string')) t = 'Walk'; else t = 'Stop'; end;
       m=0; tic; set(S.wlk,'string',t);  % toggle Walk/Stop 
       while ishandle(S.wlk) & findstr('S',get(S.wlk,'string'))
             walk(S.tr(2)); m = m+1;   % walk until button pressed again
               if ~mod(m,25) set(S.ups,'str',sprintf('%.2f',m/toc)); end;  drawnow;
       end;
  end;
% end function btn

function walk(h,r) % set the y data for line h (rand button use 2 argument form)
  y = imag(vdata)';                              % compute a random y vector
  if nargin<2                                    % for walk button
    y = 1000 - abs(1000 - abs(get(h,'y') + y/20 - 25));  y(1:3:end) = 0;
  end;
  set(h,'y',y);                                  % update line data
  if isempty(getappdata(gcf,'cid')) return; end; % in case it's closed via demoplt
  cur;                                           % update xstring
% end function walk

function v = vdata()       % generate some data to plot
  x = 1:40;                % plot 40 bars
  y = 1000*rand(size(x));  % some random data to plot
  v = Pvbar(x,0,y);        % make vertical bars
% end function vdata